# Funcionalidad de Asignación de Almacenes a Usuarios

## Descripción General
Esta funcionalidad permite asignar múltiples almacenes a usuarios cuyo rol requiere acceso a almacenes específicos. La asignación es dinámica y se valida automáticamente según la configuración del rol.

## Cambios Realizados

### 1. Base de Datos

#### Nueva Tabla: `usuario_almacenes`
**Archivo:** `create_usuario_almacenes_table.sql`

Esta tabla establece la relación many-to-many entre usuarios y almacenes.

**Estructura:**
```sql
- id: INT (PK, AUTO_INCREMENT)
- usuario_id: INT (FK → usuarios.id)
- almacen_id: INT (FK → almacenes.id)
- fecha_asignacion: TIMESTAMP
```

**Características:**
- Constraint único para evitar duplicados: `unique_usuario_almacen`
- Foreign keys con CASCADE en DELETE y UPDATE
- Índices optimizados para búsquedas rápidas

#### Cómo Ejecutar la Migración

1. **Opción 1 - MySQL CLI:**
```bash
mysql -u root -p nombre_database < create_usuario_almacenes_table.sql
```

2. **Opción 2 - phpMyAdmin:**
   - Acceder a phpMyAdmin
   - Seleccionar la base de datos
   - Ir a la pestaña "SQL"
   - Copiar y pegar el contenido del archivo
   - Ejecutar

3. **Opción 3 - MySQL Workbench:**
   - Abrir el archivo en MySQL Workbench
   - Ejecutar el script

### 2. Backend (API PHP)

#### Archivo Modificado: `UsuariosNoAuth.php`

**Cambios en `getUsuarios`:**
- Ahora incluye el campo `necesita_almacen` del rol
- Obtiene los almacenes asignados a cada usuario
- Retorna arrays `almacenes` y `almacenes_ids`

**Ejemplo de respuesta:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "usuario": "juan_almacen",
      "nombre": "Juan Pérez",
      "rol_nombre": "Almacenista",
      "necesita_almacen": true,
      "almacenes": [
        {
          "id": 1,
          "nombre": "Almacén Principal",
          "codigo": "ALM-01"
        },
        {
          "id": 2,
          "nombre": "Almacén Sucursal",
          "codigo": "ALM-02"
        }
      ],
      "almacenes_ids": [1, 2]
    }
  ]
}
```

**Cambios en `crearUsuario`:**
- Recibe parámetro `almacenes_ids` (array)
- Valida que si el rol requiere almacén, se envíe al menos uno
- Usa transacciones para garantizar consistencia
- Asigna almacenes en tabla `usuario_almacenes`

**Validaciones:**
```php
if ($rol_data['necesita_almacen'] && empty($almacenes_ids)) {
    sendResponse(false, "Este rol requiere al menos un almacén asignado");
}
```

**Cambios en `actualizarUsuario`:**
- Recibe parámetro `almacenes_ids` (array, opcional)
- Si se envía `almacenes_ids`, actualiza las asignaciones
- Elimina asignaciones anteriores y crea nuevas (replace strategy)
- Usa transacciones para operación atómica

### 3. Frontend (React)

#### Archivo Modificado: `UserForm.js`

**Nuevas importaciones:**
```javascript
import { Warehouse } from 'lucide-react'; // Icono de almacén
import './UserForm.css'; // Estilos del formulario
```

**Nuevos estados:**
```javascript
almacenes_ids: []          // IDs de almacenes seleccionados
[almacenes, setAlmacenes]  // Lista de almacenes disponibles
[loadingAlmacenes, ...]    // Estado de carga
```

**Nuevas funciones:**
- `loadAlmacenes()`: Carga almacenes disponibles de la API
- `handleAlmacenToggle(id)`: Toggle checkbox de almacenes
- `rolRequiereAlmacen()`: Verifica si rol seleccionado requiere almacén

**Validación:**
```javascript
if (rolRequiereAlmacen() && formData.almacenes_ids.length === 0) {
    newErrors.almacenes_ids = 'Este rol requiere al menos un almacén asignado';
}
```

**Nuevo componente UI:**
El selector de almacenes solo se muestra cuando `rolRequiereAlmacen()` retorna `true`.

Características:
- Checkboxes múltiples
- Muestra nombre y código del almacén
- Scroll automático si hay muchos almacenes
- Estados de carga y vacío
- Validación visual de errores

#### Archivo Creado: `UserForm.css`

Estilos específicos para:
- Layout del formulario (grid 2 columnas)
- Selector de almacenes con checkboxes
- Estados hover, focus, error
- Responsive design (1 columna en móviles)

## Flujo de Uso

### Crear Usuario con Almacenes

1. Usuario abre el modal "Crear Nuevo Usuario"
2. Llena datos básicos (usuario, nombre, email, etc.)
3. Selecciona un rol
4. Si el rol tiene `necesita_almacen = true`:
   - Aparece selector de almacenes
   - Debe seleccionar al menos uno
5. Al enviar:
   - Frontend envía `almacenes_ids: [1, 2, 3]`
   - Backend valida y crea usuario
   - Backend asigna almacenes en transacción
6. Usuario creado exitosamente

### Editar Usuario y Cambiar Almacenes

1. Usuario abre modal "Editar Usuario"
2. Selector muestra almacenes previamente asignados (checked)
3. Puede agregar/quitar almacenes
4. Al guardar:
   - Backend elimina asignaciones anteriores
   - Backend crea nuevas asignaciones
   - Todo en una transacción

### Cambiar Rol que No Requiere Almacén

1. Si se cambia de rol que requiere almacén a uno que no:
   - Selector desaparece
   - `almacenes_ids` se envía vacío
   - Backend elimina asignaciones
2. Si se cambia a rol que sí requiere:
   - Selector aparece
   - Debe seleccionar almacenes

## Estructura de Datos

### Request para Crear Usuario
```json
{
  "tipo": "crearUsuario",
  "usuario": "juan_almacen",
  "nombre": "Juan Pérez",
  "email": "juan@empresa.com",
  "password": "password123",
  "rol_id": 4,
  "telefono": "5551234567",
  "almacenes_ids": [1, 2]
}
```

### Request para Actualizar Usuario
```json
{
  "tipo": "actualizarUsuario",
  "user_id": 5,
  "nombre": "Juan Pérez García",
  "email": "juan.perez@empresa.com",
  "rol_id": 4,
  "telefono": "5551234567",
  "activo": 1,
  "almacenes_ids": [1, 2, 3]  // Opcional
}
```

### Response de getUsuarios
```json
{
  "success": true,
  "message": "Usuarios obtenidos",
  "data": [
    {
      "id": 5,
      "usuario": "juan_almacen",
      "nombre": "Juan Pérez",
      "email": "juan@empresa.com",
      "rol_id": 4,
      "rol_nombre": "Almacenista",
      "necesita_almacen": true,
      "almacenes": [
        { "id": 1, "nombre": "Almacén Principal", "codigo": "ALM-01" },
        { "id": 2, "nombre": "Almacén Sucursal", "codigo": "ALM-02" }
      ],
      "almacenes_ids": [1, 2],
      "activo": 1,
      "ultimo_acceso": "2025-11-08 10:30:00"
    }
  ]
}
```

## Validaciones

### Backend
1. ✅ Rol requiere almacén pero no se envió ninguno → Error
2. ✅ IDs de almacenes inválidos son filtrados
3. ✅ Uso de transacciones para garantizar consistencia
4. ✅ Foreign keys previenen asignar almacenes inexistentes
5. ✅ Unique constraint previene duplicados

### Frontend
1. ✅ Selector solo visible si rol lo requiere
2. ✅ Validación antes de submit
3. ✅ Mensajes de error claros
4. ✅ Estados de carga durante fetch
5. ✅ Manejo de casos sin almacenes disponibles

## Consultas SQL Útiles

### Ver asignaciones de un usuario
```sql
SELECT u.nombre, a.nombre as almacen, a.codigo
FROM usuarios u
JOIN usuario_almacenes ua ON u.id = ua.usuario_id
JOIN almacenes a ON ua.almacen_id = a.id
WHERE u.id = ?;
```

### Ver usuarios con acceso a un almacén
```sql
SELECT u.nombre, u.usuario, r.nombre as rol
FROM usuarios u
JOIN usuario_almacenes ua ON u.id = ua.usuario_id
JOIN roles r ON u.rol_id = r.id
WHERE ua.almacen_id = ? AND u.activo = 1;
```

### Ver usuarios sin almacenes asignados pero con rol que lo requiere
```sql
SELECT u.id, u.nombre, r.nombre as rol
FROM usuarios u
JOIN roles r ON u.rol_id = r.id
LEFT JOIN usuario_almacenes ua ON u.id = ua.usuario_id
WHERE r.necesita_almacen = 1
  AND ua.usuario_id IS NULL
  AND u.activo = 1;
```

## Archivos Modificados

### Backend
- `C:\xampp4\htdocs\api_pos\reportes\funciones_php\UsuariosNoAuth.php`
- `C:\xampp4\htdocs\api_pos\database\migrations\create_usuario_almacenes_table.sql` (nuevo)

### Frontend
- `C:\xampp\htdocs\abarrotesv2\pos_abarrotes\src\pages\users\UserForm.js`
- `C:\xampp\htdocs\abarrotesv2\pos_abarrotes\src\pages\users\UserForm.css` (nuevo)

## Próximos Pasos Recomendados

1. **Seguridad:**
   - Implementar autenticación en endpoints (actualmente usa `UsuariosNoAuth.php`)
   - Verificar permisos del usuario logueado antes de asignar almacenes

2. **Funcionalidad:**
   - Mostrar almacenes asignados en `UserDetails.js`
   - Agregar filtro en `UsersPage.js` por almacén
   - Restringir vistas/acciones según almacén del usuario

3. **Reportes:**
   - Reporte de usuarios por almacén
   - Auditoría de cambios en asignaciones

4. **Testing:**
   - Probar creación con diferentes roles
   - Probar edición cambiando de rol
   - Verificar integridad referencial al eliminar almacenes

## Soporte

Para dudas o problemas:
1. Verificar logs de errores en PHP
2. Revisar consola del navegador para errores JS
3. Verificar que la tabla `usuario_almacenes` existe
4. Confirmar que roles tienen campo `necesita_almacen` configurado

---

**Fecha de implementación:** 2025-11-08
**Versión:** 1.0 - Selector condicional basado en necesita_almacen

**Importante:** El selector de almacenes SOLO aparece en el formulario cuando el rol del usuario tiene `necesita_almacen = 1`
