<?php
/**
 * Script para recuperar datos de la tabla pedido corrupta
 * Lee el archivo .MYD y extrae los datos legibles
 */

$mydFile = 'C:/xampp4/mysql/data/kapital_restaurant_queretaro/pedido.MYD.backup_20251101_202437';
$csvFile = 'C:/xampp4/htdocs/api_pos/pedidos_recuperados.csv';
$sqlFile = 'C:/xampp4/htdocs/api_pos/pedidos_recuperados.sql';

// Estructura de la tabla pedido según base1
$campos = [
    'idPedido', 'numero', 'numeroDia', 'numeroPersonas', 'idCliente',
    'fechaCreacion', 'mesero', 'tipo', 'numeroTipo', 'horaAbierto',
    'horaCerrado', 'horaEnvio', 'status', 'numeroImpresiones', 'usuario',
    'liquidado', 'numeroTicket', 'supervisorCancela', 'comentarios',
    'dividida', 'clienteLlevar', 'banderaDescuento', 'valorDescuento',
    'horaSalida', 'enviadoServicio', 'codigopedidofactura',
    'fecha_hora_salioReparto', 'fecha_hora_regresoReparto',
    'fecha_hora_salioCocina', 'facturado', 'Sonido', 'coordenadas',
    'facturadoGeneral', 'referenciaFactura'
];

echo "Leyendo archivo .MYD...\n";
$contenido = file_get_contents($mydFile);
$tamano = strlen($contenido);
echo "Tamaño del archivo: " . number_format($tamano) . " bytes\n\n";

// Patrones comunes encontrados en los datos
$registros = [];
$idCounter = 1;

// Extraer todo el texto legible primero
$textoCompleto = '';
for ($i = 0; $i < $tamano; $i++) {
    $char = $contenido[$i];
    $ord = ord($char);
    if ($ord >= 32 && $ord <= 126) {
        $textoCompleto .= $char;
    } else if ($ord == 10 || $ord == 13) {
        $textoCompleto .= "\n";
    } else {
        $textoCompleto .= ' ';
    }
}

echo "Buscando patrones de pedidos...\n";

// Patrón mejorado: buscar secuencias de texto que contengan mesero + tipo + status
preg_match_all('/([A-Z]{3,}?)(mesa|llevar|delivery).*?(cerrado|cancelado|abierto).*?(soporte1?|lizeth|dayana)/i', $textoCompleto, $matches, PREG_SET_ORDER);

echo "Encontrados " . count($matches) . " patrones iniciales\n";

foreach ($matches as $match) {
    $registro = [];
    $registro['idPedido'] = $idCounter++;
    $registro['mesero'] = trim($match[1]);
    $registro['tipo'] = trim($match[2]);
    $registro['status'] = trim($match[3]);
    $registro['usuario'] = trim($match[4]);
    $registro['clienteLlevar'] = '';
    $registro['comentarios'] = '';
    $registro['supervisorCancela'] = '';
    $registro['codigopedidofactura'] = '';

    // Extraer contexto alrededor del match para más detalles
    $contexto = $match[0];

    // Buscar supervisor
    if (preg_match('/Abdiel Najar/', $contexto, $m)) {
        $registro['supervisorCancela'] = 'Abdiel Najar';
    }

    // Buscar teléfono (10 dígitos)
    if (preg_match('/(\d{10})/', $contexto, $m)) {
        $registro['codigopedidofactura'] = $m[1];
    }

    // Buscar comentarios
    if (preg_match('/(RAPPI ID\d+|rappi id\d+|comida personal|PRUEBA COMANDAS|COMIDA DE AGS|CUMPLEAERO|la chapu|sr wings)/i', $contexto, $m)) {
        $registro['comentarios'] = $m[1];
    }

    // Buscar nombre de cliente
    if (preg_match('/([A-Z][a-z]+)\s*\d{10}/', $contexto, $m)) {
        $registro['clienteLlevar'] = $m[1];
    }

    $registros[] = $registro;
}

// Segunda pasada: buscar por patrón más simple
echo "Segunda pasada con patrón simplificado...\n";
$lineas = explode("\n", $textoCompleto);
foreach ($lineas as $linea) {
    if (strlen($linea) > 20 &&
        (stripos($linea, 'mesa') !== false || stripos($linea, 'llevar') !== false || stripos($linea, 'delivery') !== false) &&
        (stripos($linea, 'cerrado') !== false || stripos($linea, 'cancelado') !== false)) {

        $registro = [];
        $registro['idPedido'] = $idCounter++;
        $registro['mesero'] = '';
        $registro['tipo'] = '';
        $registro['status'] = '';
        $registro['usuario'] = '';
        $registro['clienteLlevar'] = '';
        $registro['comentarios'] = '';
        $registro['supervisorCancela'] = '';
        $registro['codigopedidofactura'] = '';

        // Extraer mesero (nombre en mayúsculas)
        if (preg_match('/([A-Z]{4,})/', $linea, $m)) {
            $registro['mesero'] = $m[1];
        }

        // Extraer tipo
        if (preg_match('/(mesa|llevar|delivery)/i', $linea, $m)) {
            $registro['tipo'] = $m[1];
        }

        // Extraer status
        if (preg_match('/(cerrado|cancelado)/i', $linea, $m)) {
            $registro['status'] = $m[1];
        }

        // Extraer usuario
        if (preg_match('/(soporte1?|lizeth|dayana)/i', $linea, $m)) {
            $registro['usuario'] = $m[1];
        }

        // Extraer teléfono
        if (preg_match('/(\d{10})/', $linea, $m)) {
            $registro['codigopedidofactura'] = $m[1];
        }

        // Buscar supervisor
        if (preg_match('/Abdiel Najar/', $linea)) {
            $registro['supervisorCancela'] = 'Abdiel Najar';
        }

        // Solo agregar si tiene datos mínimos
        if (!empty($registro['tipo']) && !empty($registro['status'])) {
            $registros[] = $registro;
        }
    }
}

echo "Total de registros encontrados: " . count($registros) . "\n\n";

// Generar CSV
echo "Generando archivo CSV...\n";
$fp = fopen($csvFile, 'w');
fputcsv($fp, ['idPedido', 'mesero', 'tipo', 'status', 'usuario', 'clienteLlevar', 'comentarios', 'supervisorCancela', 'codigopedidofactura']);

foreach ($registros as $reg) {
    fputcsv($fp, [
        $reg['idPedido'],
        $reg['mesero'],
        $reg['tipo'],
        $reg['status'],
        $reg['usuario'],
        $reg['clienteLlevar'],
        $reg['comentarios'],
        $reg['supervisorCancela'],
        $reg['codigopedidofactura']
    ]);
}
fclose($fp);
echo "Archivo CSV generado: $csvFile\n\n";

// Generar SQL
echo "Generando archivo SQL...\n";
$sql = "-- Datos recuperados de la tabla pedido corrupta\n";
$sql .= "-- Fecha de recuperación: " . date('Y-m-d H:i:s') . "\n";
$sql .= "-- Total de registros recuperados: " . count($registros) . "\n\n";
$sql .= "USE kapital_restaurant_queretaro;\n\n";

foreach ($registros as $reg) {
    $mesero = addslashes($reg['mesero']);
    $tipo = addslashes($reg['tipo']);
    $status = addslashes($reg['status']);
    $usuario = addslashes($reg['usuario']);
    $clienteLlevar = addslashes($reg['clienteLlevar']);
    $comentarios = addslashes($reg['comentarios']);
    $supervisorCancela = addslashes($reg['supervisorCancela']);
    $codigopedidofactura = addslashes($reg['codigopedidofactura']);

    $sql .= "INSERT INTO pedido (numero, numeroDia, numeroPersonas, idCliente, mesero, tipo, status, usuario, clienteLlevar, comentarios, supervisorCancela, codigopedidofactura, numeroTipo, numeroImpresiones, liquidado, numeroTicket, dividida, banderaDescuento, valorDescuento, enviadoServicio, facturado, Sonido, facturadoGeneral) VALUES\n";
    $sql .= "(0, 0, 0, 0, '$mesero', '$tipo', '$status', '$usuario', '$clienteLlevar', '$comentarios', '$supervisorCancela', '$codigopedidofactura', 0, 0, '0', 0, '0', 0, 0, 0, 0, 0, 0);\n";
}

file_put_contents($sqlFile, $sql);
echo "Archivo SQL generado: $sqlFile\n\n";

// Mostrar muestra de los datos
echo "Muestra de los primeros 10 registros:\n";
echo str_repeat("-", 100) . "\n";
printf("%-5s %-15s %-12s %-12s %-12s %-20s\n", "ID", "Mesero", "Tipo", "Status", "Usuario", "Cliente/Tel");
echo str_repeat("-", 100) . "\n";

for ($i = 0; $i < min(10, count($registros)); $i++) {
    $reg = $registros[$i];
    printf("%-5d %-15s %-12s %-12s %-12s %-20s\n",
        $reg['idPedido'],
        substr($reg['mesero'], 0, 15),
        substr($reg['tipo'], 0, 12),
        substr($reg['status'], 0, 12),
        substr($reg['usuario'], 0, 12),
        substr($reg['clienteLlevar'] ?: $reg['codigopedidofactura'], 0, 20)
    );
}

echo str_repeat("-", 100) . "\n";
echo "\nArchivos generados exitosamente!\n";
echo "CSV: $csvFile\n";
echo "SQL: $sqlFile\n";
?>
