<?php
/**
 * Script mejorado para recuperar datos de la tabla pedido corrupta
 * Versión 2: Incluye extracción de fechas y campos numéricos
 */

$mydFile = 'C:/xampp4/mysql/data/kapital_restaurant_queretaro/pedido.MYD.backup_20251101_202437';
$csvFile = 'C:/xampp4/htdocs/api_pos/pedidos_recuperados_v2.csv';
$sqlFile = 'C:/xampp4/htdocs/api_pos/pedidos_recuperados_v2.sql';

echo "Leyendo archivo .MYD...\n";
$handle = fopen($mydFile, 'rb');
$tamano = filesize($mydFile);
echo "Tamaño del archivo: " . number_format($tamano) . " bytes\n\n";

$registros = [];
$offset = 0;
$recordSize = 150; // Tamaño aproximado de cada registro

echo "Extrayendo registros...\n";

while ($offset < $tamano - $recordSize) {
    fseek($handle, $offset);
    $chunk = fread($handle, $recordSize);

    // Buscar inicio de registro válido (contiene texto de mesero o tipo)
    if (preg_match('/(RENATA|ALEJANDRO|DANA|DAYANA|ABDIEL|DANIEL|MANU|JOHAN|EUNICE|IRMA|ISAAC|VACANTE|RAPPI|UBER|GUERRERO)/i', $chunk, $meseroMatch) &&
        preg_match('/(mesa|llevar|delivery)/i', $chunk, $tipoMatch) &&
        preg_match('/(cerrado|cancelado|abierto)/i', $chunk, $statusMatch)) {

        $registro = [];

        // Extraer texto legible
        $textoLegible = '';
        for ($i = 0; $i < strlen($chunk); $i++) {
            $char = $chunk[$i];
            if (ord($char) >= 32 && ord($char) <= 126) {
                $textoLegible .= $char;
            } else {
                $textoLegible .= ' ';
            }
        }

        // Intentar extraer fecha (3 bytes en formato MySQL date)
        // MySQL almacena fechas como: ((year-1900)*16*32 + month*32 + day)
        $fechaPos = strpos($chunk, chr(0x90).chr(0xcd).chr(0x0f)); // Patrón común encontrado
        if ($fechaPos !== false && $fechaPos >= 10) {
            // Leer 3 bytes antes del patrón
            $dateByte1 = ord($chunk[$fechaPos]);
            $dateByte2 = ord($chunk[$fechaPos + 1]);
            $dateByte3 = ord($chunk[$fechaPos + 2]);

            // Decodificar fecha MySQL
            $dateInt = ($dateByte1 << 16) | ($dateByte2 << 8) | $dateByte3;
            $day = $dateInt & 31;
            $month = ($dateInt >> 5) & 15;
            $year = ($dateInt >> 9);

            if ($year > 0 && $month > 0 && $month <= 12 && $day > 0 && $day <= 31) {
                $registro['fechaCreacion'] = sprintf('%04d-%02d-%02d', 1900 + $year, $month, $day);
            } else {
                $registro['fechaCreacion'] = null;
            }
        } else {
            $registro['fechaCreacion'] = null;
        }

        // Extraer número de pedido (primeros 4 bytes del registro)
        $numBytes = substr($chunk, 3, 4);
        if (strlen($numBytes) == 4) {
            $numPedido = unpack('V', $numBytes)[1]; // Unsigned long little-endian
            if ($numPedido > 0 && $numPedido < 1000000) {
                $registro['numero'] = $numPedido;
            } else {
                $registro['numero'] = 0;
            }
        } else {
            $registro['numero'] = 0;
        }

        // Extraer campos de texto
        $registro['mesero'] = trim($meseroMatch[1]);
        $registro['tipo'] = trim($tipoMatch[1]);
        $registro['status'] = trim($statusMatch[1]);

        // Usuario
        if (preg_match('/(soporte1?|lizeth|dayana)/i', $textoLegible, $m)) {
            $registro['usuario'] = trim($m[1]);
        } else {
            $registro['usuario'] = '';
        }

        // Cliente/Teléfono
        if (preg_match('/(\d{10})/', $textoLegible, $m)) {
            $registro['telefono'] = $m[1];
        } else {
            $registro['telefono'] = '';
        }

        // Supervisor
        if (preg_match('/Abdiel Najar/', $textoLegible)) {
            $registro['supervisorCancela'] = 'Abdiel Najar';
        } else {
            $registro['supervisorCancela'] = '';
        }

        // Comentarios
        $comentarios = [];
        if (preg_match('/(RAPPI ID\d+|rappi id\d+)/i', $textoLegible, $m)) {
            $comentarios[] = $m[1];
        }
        if (preg_match('/(comida personal|PRUEBA COMANDAS|COMIDA DE AGS|CUMPLEAERO|la chapu|sr wings)/i', $textoLegible, $m)) {
            $comentarios[] = $m[1];
        }
        $registro['comentarios'] = implode(', ', $comentarios);

        // Cliente para llevar
        if ($registro['tipo'] == 'llevar' && preg_match('/([A-Z][a-z]{2,})/i', $textoLegible, $m)) {
            $registro['clienteLlevar'] = $m[1];
        } else {
            $registro['clienteLlevar'] = '';
        }

        $registros[] = $registro;
        $offset += 80; // Avanzar menos para no perder registros
    } else {
        $offset += 20; // Avanzar poco a poco
    }

    // Mostrar progreso
    if (count($registros) % 1000 == 0 && count($registros) > 0) {
        $progreso = ($offset / $tamano) * 100;
        echo sprintf("Progreso: %.1f%% - Registros encontrados: %d\r", $progreso, count($registros));
    }
}

fclose($handle);
echo "\nTotal de registros encontrados: " . count($registros) . "\n\n";

// Generar CSV
echo "Generando archivo CSV...\n";
$fp = fopen($csvFile, 'w');
fputcsv($fp, ['numero', 'fechaCreacion', 'mesero', 'tipo', 'status', 'usuario', 'telefono', 'clienteLlevar', 'comentarios', 'supervisorCancela']);

foreach ($registros as $reg) {
    fputcsv($fp, [
        $reg['numero'],
        $reg['fechaCreacion'],
        $reg['mesero'],
        $reg['tipo'],
        $reg['status'],
        $reg['usuario'],
        $reg['telefono'],
        $reg['clienteLlevar'],
        $reg['comentarios'],
        $reg['supervisorCancela']
    ]);
}
fclose($fp);
echo "Archivo CSV generado: $csvFile\n\n";

// Generar SQL
echo "Generando archivo SQL...\n";
$sql = "-- Datos recuperados de la tabla pedido corrupta (Versión 2 con fechas)\n";
$sql .= "-- Fecha de recuperación: " . date('Y-m-d H:i:s') . "\n";
$sql .= "-- Total de registros recuperados: " . count($registros) . "\n\n";
$sql .= "USE kapital_restaurant_queretaro;\n\n";

foreach ($registros as $reg) {
    $numero = intval($reg['numero']);
    $fecha = $reg['fechaCreacion'] ? "'" . addslashes($reg['fechaCreacion']) . "'" : "NULL";
    $mesero = addslashes($reg['mesero']);
    $tipo = addslashes($reg['tipo']);
    $status = addslashes($reg['status']);
    $usuario = addslashes($reg['usuario']);
    $telefono = addslashes($reg['telefono']);
    $clienteLlevar = addslashes($reg['clienteLlevar']);
    $comentarios = addslashes($reg['comentarios']);
    $supervisorCancela = addslashes($reg['supervisorCancela']);

    $sql .= "INSERT INTO pedido (numero, numeroDia, numeroPersonas, idCliente, fechaCreacion, mesero, tipo, status, usuario, clienteLlevar, comentarios, supervisorCancela, codigopedidofactura, numeroTipo, numeroImpresiones, liquidado, numeroTicket, dividida, banderaDescuento, valorDescuento, enviadoServicio, facturado, Sonido, facturadoGeneral) VALUES\n";
    $sql .= "($numero, 0, 0, 0, $fecha, '$mesero', '$tipo', '$status', '$usuario', '$clienteLlevar', '$comentarios', '$supervisorCancela', '$telefono', 0, 0, '0', 0, '0', 0, 0, 0, 0, 0, 0);\n";
}

file_put_contents($sqlFile, $sql);
echo "Archivo SQL generado: $sqlFile\n\n";

// Estadísticas
echo "=== ESTADÍSTICAS DE RECUPERACIÓN ===\n";
echo str_repeat("-", 50) . "\n";

// Registros con fecha
$conFecha = array_filter($registros, function($r) { return $r['fechaCreacion'] !== null; });
echo "Registros con fecha: " . count($conFecha) . " (" . round(count($conFecha)/count($registros)*100, 1) . "%)\n";

// Por tipo
$tipos = array_count_values(array_column($registros, 'tipo'));
arsort($tipos);
echo "\nPor tipo de pedido:\n";
foreach ($tipos as $tipo => $count) {
    echo "  - " . ucfirst($tipo) . ": " . number_format($count) . "\n";
}

// Por status
$status = array_count_values(array_column($registros, 'status'));
arsort($status);
echo "\nPor estado:\n";
foreach ($status as $st => $count) {
    echo "  - " . ucfirst($st) . ": " . number_format($count) . "\n";
}

// Top meseros
$meseros = array_count_values(array_column($registros, 'mesero'));
arsort($meseros);
echo "\nTop 10 meseros:\n";
$i = 1;
foreach (array_slice($meseros, 0, 10, true) as $mesero => $count) {
    echo "  $i. $mesero: " . number_format($count) . " pedidos\n";
    $i++;
}

// Muestra de datos
echo "\n=== MUESTRA DE LOS PRIMEROS 10 REGISTROS ===\n";
echo str_repeat("-", 120) . "\n";
printf("%-8s %-12s %-12s %-10s %-12s %-12s %-15s\n", "Número", "Fecha", "Mesero", "Tipo", "Status", "Usuario", "Teléfono");
echo str_repeat("-", 120) . "\n";

for ($i = 0; $i < min(10, count($registros)); $i++) {
    $reg = $registros[$i];
    printf("%-8s %-12s %-12s %-10s %-12s %-12s %-15s\n",
        $reg['numero'] ?: 'N/A',
        $reg['fechaCreacion'] ?: 'N/A',
        substr($reg['mesero'], 0, 12),
        substr($reg['tipo'], 0, 10),
        substr($reg['status'], 0, 12),
        substr($reg['usuario'], 0, 12),
        substr($reg['telefono'], 0, 15)
    );
}

echo str_repeat("-", 120) . "\n";
echo "\n✓ Archivos generados exitosamente!\n";
echo "  CSV: $csvFile\n";
echo "  SQL: $sqlFile\n";
?>
