<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'login':
    case 'authenticateUser':
        $usuario = isset($datapost['usuario']) ? $security->validateInput($datapost['usuario']) : '';
        $password = isset($datapost['password']) ? $datapost['password'] : '';
        $biometric_id = isset($datapost['biometric_id']) ? $datapost['biometric_id'] : null;

        if (empty($usuario) && empty($biometric_id)) {
            sendResponse(false, "Usuario o ID biométrico requerido");
        }

        // Verificar bloqueo por intentos fallidos
        $identifier = !empty($usuario) ? $usuario : $biometric_id;
        $bruteForceCheck = $security->checkBruteForce($identifier);

        if ($bruteForceCheck['blocked']) {
            $security->logSecurityEvent('login_blocked', null, "Usuario bloqueado: $identifier");
            sendResponse(false, $bruteForceCheck['message']);
        }

        try {
            if (!empty($biometric_id)) {
                // Autenticación biométrica
                $query = "SELECT u.*, r.nombre as rol_nombre
                         FROM usuarios u
                         LEFT JOIN roles r ON u.rol_id = r.id
                         WHERE u.biometric_id = ? AND u.activo = 1";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("s", $biometric_id);
            } else {
                // Autenticación por usuario/password
                $query = "SELECT u.*, r.nombre as rol_nombre
                         FROM usuarios u
                         LEFT JOIN roles r ON u.rol_id = r.id
                         WHERE (u.usuario = ? OR u.email = ?) AND u.activo = 1";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ss", $usuario, $usuario);
            }

            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();

            if (!$user) {
                $security->recordFailedAttempt($identifier);
                $security->logSecurityEvent('login_failed', null, "Usuario no encontrado: $identifier");
                sendResponse(false, "Credenciales incorrectas1");
            }

            // Verificar password (solo para autenticación no biométrica)
            if (!empty($password) && !$myencrypt->verifyPassword($password, $user['password'])) {
                $security->recordFailedAttempt($identifier);
                $security->logSecurityEvent('login_failed', $user['id'], "Password incorrecto");
                sendResponse(false, "Credenciales incorrectas2 ");
            }

            // Obtener permisos del rol
            $permisosQuery = "SELECT p.nombre
                             FROM rol_permisos rp
                             JOIN permisos p ON rp.permiso_id = p.id
                             WHERE rp.rol_id = ? AND p.activo = 1";
            $permisosStmt = $conn->prepare($permisosQuery);
            $permisosStmt->bind_param("i", $user['rol_id']);
            $permisosStmt->execute();
            $permisosResult = $permisosStmt->get_result();

            $permisos = [];
            while ($row = $permisosResult->fetch_assoc()) {
                $permisos[] = $row['nombre'];
            }

            // Obtener almacenes asignados al usuario
            $almacenesQuery = "SELECT a.id, a.nombre, a.codigo
                              FROM usuario_almacenes ua
                              JOIN almacenes a ON ua.almacen_id = a.id
                              WHERE ua.usuario_id = ? AND a.activo = 1";
            $almacenesStmt = $conn->prepare($almacenesQuery);
            $almacenesStmt->bind_param("i", $user['id']);
            $almacenesStmt->execute();
            $almacenesResult = $almacenesStmt->get_result();

            $almacenes = [];
            while ($almacen = $almacenesResult->fetch_assoc()) {
                $almacenes[] = [
                    'id' => intval($almacen['id']),
                    'nombre' => $almacen['nombre'],
                    'codigo' => $almacen['codigo']
                ];
            }

            // Login exitoso
            $security->clearFailedAttempts($identifier);
            $session_token = $security->createSession($user['id'], [
                'usuario' => $user['usuario'],
                'nombre' => $user['nombre'],
                'email' => $user['email'],
                'rol' => $user['rol_nombre'],
                'permisos' => $permisos,
                'almacenes' => $almacenes
            ]);

            // Actualizar último acceso
            $updateQuery = "UPDATE usuarios SET ultimo_acceso = NOW() WHERE id = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->bind_param("i", $user['id']);
            $updateStmt->execute();

            $security->logSecurityEvent('login_success', $user['id'], "Login exitoso");

            sendResponse(true, "Login exitoso", [
                'user_id' => $user['id'],
                'usuario' => $user['usuario'],
                'nombre' => $user['nombre'],
                'email' => $user['email'],
                'rol' => $user['rol_nombre'],
                'rol_id' => $user['rol_id'],
                'permisos' => $permisos,
                'almacenes' => $almacenes,
                'session_token' => $session_token,
                'ultimo_acceso' => $user['ultimo_acceso']
            ]);
        } catch (Exception $e) {
            $security->logSecurityEvent('login_error', null, "Error en login: " . $e->getMessage());
            sendResponse(false, "Error interno del servidor");
        }
        break;

    case 'logout':
        $session = $security->validateSession();
        if ($session['valid']) {
            $security->logSecurityEvent('logout', $_SESSION['user_id'], "Logout exitoso");
            $security->destroySession();
            sendResponse(true, "Sesión cerrada exitosamente");
        } else {
            sendResponse(false, "No hay sesión activa");
        }
        break;

    case 'refreshToken':
        $session = $security->validateSession();
        if ($session['valid']) {
            $new_token = $security->createSession($_SESSION['user_id'], $_SESSION['user_data']);
            sendResponse(true, "Token renovado", ['session_token' => $new_token]);
        } else {
            sendResponse(false, $session['message']);
        }
        break;

    case 'getUsuarios':
        $session = $security->validateSession(['gestionar_usuarios']);
        if (!$session['valid']) {
            sendResponse(false, $session['message']);
        }

        try {
            $query = "SELECT u.id, u.usuario, u.nombre, u.email, u.telefono, u.activo,
                            u.fecha_creacion, u.ultimo_acceso, u.rol_id, r.nombre as rol_nombre
                     FROM usuarios u
                     LEFT JOIN roles r ON u.rol_id = r.id
                     ORDER BY u.nombre";

            $result = $conn->query($query);
            $usuarios = [];

            while ($row = $result->fetch_assoc()) {
                $usuarios[] = $row;
            }

            sendResponse(true, "Usuarios obtenidos", $usuarios);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener usuarios: " . $e->getMessage());
        }
        break;

    case 'crearUsuario':
        $session = $security->validateSession(['gestionar_usuarios']);
        if (!$session['valid']) {
            sendResponse(false, $session['message']);
        }

        $usuario = $security->validateInput($datapost['usuario'] ?? '');
        $nombre = $security->validateInput($datapost['nombre'] ?? '');
        $email = $security->validateInput($datapost['email'] ?? '');
        $password = $datapost['password'] ?? '';
        $rol_id = intval($datapost['rol_id'] ?? 0);
        $telefono = $security->validateInput($datapost['telefono'] ?? '');

        if (empty($usuario) || empty($nombre) || empty($email) || empty($password) || $rol_id == 0) {
            sendResponse(false, "Todos los campos son requeridos");
        }

        if (!validarEmail($email)) {
            sendResponse(false, "Email no válido");
        }

        try {
            // Verificar si el usuario ya existe
            $checkQuery = "SELECT id FROM usuarios WHERE usuario = ? OR email = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("ss", $usuario, $email);
            $checkStmt->execute();

            if ($checkStmt->get_result()->num_rows > 0) {
                sendResponse(false, "El usuario o email ya existe");
            }

            // Crear usuario
            $hashedPassword = $myencrypt->hashPassword($password);

            $query = "INSERT INTO usuarios (usuario, nombre, email, password, rol_id, telefono, activo, fecha_creacion)
                     VALUES (?, ?, ?, ?, ?, ?, 1, NOW())";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("ssssiss", $usuario, $nombre, $email, $hashedPassword, $rol_id, $telefono);

            if ($stmt->execute()) {
                $user_id = $conn->insert_id;
                logActivity('crear_usuario', $_SESSION['user_id'], "Usuario creado: $usuario");
                sendResponse(true, "Usuario creado exitosamente", ['user_id' => $user_id]);
            } else {
                sendResponse(false, "Error al crear usuario");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al crear usuario: " . $e->getMessage());
        }
        break;

    case 'actualizarUsuario':
        $session = $security->validateSession(['gestionar_usuarios']);
        if (!$session['valid']) {
            sendResponse(false, $session['message']);
        }

        $user_id = intval($datapost['user_id'] ?? 0);
        $nombre = $security->validateInput($datapost['nombre'] ?? '');
        $email = $security->validateInput($datapost['email'] ?? '');
        $rol_id = intval($datapost['rol_id'] ?? 0);
        $telefono = $security->validateInput($datapost['telefono'] ?? '');
        $activo = intval($datapost['activo'] ?? 1);
        $password = $datapost['password'] ?? '';

        if ($user_id == 0 || empty($nombre) || empty($email) || $rol_id == 0) {
            sendResponse(false, "Datos requeridos faltantes");
        }

        if (!validarEmail($email)) {
            sendResponse(false, "Email no válido");
        }

        try {
            // Verificar si el email ya está en uso por otro usuario
            $checkQuery = "SELECT id FROM usuarios WHERE email = ? AND id != ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("si", $email, $user_id);
            $checkStmt->execute();

            if ($checkStmt->get_result()->num_rows > 0) {
                sendResponse(false, "El email ya está en uso por otro usuario");
            }

            // Si se proporciona password, actualizar con password
            if (!empty($password)) {
                $hashedPassword = $myencrypt->hashPassword($password);
                $query = "UPDATE usuarios SET nombre = ?, email = ?, rol_id = ?, telefono = ?, activo = ?, password = ?
                         WHERE id = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ssisssi", $nombre, $email, $rol_id, $telefono, $activo, $hashedPassword, $user_id);
            } else {
                // Actualizar sin cambiar el password
                $query = "UPDATE usuarios SET nombre = ?, email = ?, rol_id = ?, telefono = ?, activo = ?
                         WHERE id = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ssissi", $nombre, $email, $rol_id, $telefono, $activo, $user_id);
            }

            if ($stmt->execute()) {
                $mensaje = !empty($password) ? "Usuario actualizado (contraseña cambiada) ID: $user_id" : "Usuario actualizado ID: $user_id";
                logActivity('actualizar_usuario', $_SESSION['user_id'], $mensaje);
                sendResponse(true, "Usuario actualizado exitosamente");
            } else {
                sendResponse(false, "Error al actualizar usuario");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al actualizar usuario: " . $e->getMessage());
        }
        break;

    case 'cambiarPassword':
        $session = $security->validateSession();
        if (!$session['valid']) {
            sendResponse(false, $session['message']);
        }

        $user_id = isset($datapost['user_id']) ? intval($datapost['user_id']) : $_SESSION['user_id'];
        $password_actual = $datapost['password_actual'] ?? '';
        $password_nueva = $datapost['password_nueva'] ?? '';

        // Solo administradores pueden cambiar password de otros usuarios
        if ($user_id != $_SESSION['user_id']) {
            $adminCheck = $security->validateSession(['gestionar_usuarios']);
            if (!$adminCheck['valid']) {
                sendResponse(false, "No tienes permisos para cambiar la contraseña de otro usuario");
            }
        } else {
            // Si es el mismo usuario, verificar password actual
            if (empty($password_actual)) {
                sendResponse(false, "Contraseña actual requerida");
            }
        }

        if (empty($password_nueva)) {
            sendResponse(false, "Nueva contraseña requerida");
        }

        try {
            // Verificar password actual si es necesario
            if ($user_id == $_SESSION['user_id']) {
                $checkQuery = "SELECT password FROM usuarios WHERE id = ?";
                $checkStmt = $conn->prepare($checkQuery);
                $checkStmt->bind_param("i", $user_id);
                $checkStmt->execute();
                $result = $checkStmt->get_result();
                $userData = $result->fetch_assoc();

                if (!$myencrypt->verifyPassword($password_actual, $userData['password'])) {
                    sendResponse(false, "Contraseña actual incorrecta");
                }
            }

            // Actualizar contraseña
            $hashedPassword = $myencrypt->hashPassword($password_nueva);
            $query = "UPDATE usuarios SET password = ? WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("si", $hashedPassword, $user_id);

            if ($stmt->execute()) {
                logActivity('cambiar_password', $_SESSION['user_id'], "Password cambiado para usuario ID: $user_id");
                sendResponse(true, "Contraseña cambiada exitosamente");
            } else {
                sendResponse(false, "Error al cambiar contraseña");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al cambiar contraseña: " . $e->getMessage());
        }
        break;

    case 'eliminarUsuario':
        $session = $security->validateSession(['gestionar_usuarios']);
        if (!$session['valid']) {
            sendResponse(false, $session['message']);
        }

        $user_id = intval($datapost['user_id'] ?? 0);

        if ($user_id == 0) {
            sendResponse(false, "ID de usuario requerido");
        }

        if ($user_id == $_SESSION['user_id']) {
            sendResponse(false, "No puedes eliminar tu propio usuario");
        }

        try {
            $query = "UPDATE usuarios SET activo = 0 WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $user_id);

            if ($stmt->execute()) {
                logActivity('eliminar_usuario', $_SESSION['user_id'], "Usuario eliminado ID: $user_id");
                sendResponse(true, "Usuario eliminado exitosamente");
            } else {
                sendResponse(false, "Error al eliminar usuario");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al eliminar usuario: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación de autenticación no válida");
        break;
}
