<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'getClientes':
        try {
            // Query con las nuevas tablas relacionadas
            $query = "SELECT
                c.id,
                c.nombre,
                c.nombre_comercial,
                c.precio_venta,
                c.activo,
                c.tiene_credito,
                c.monto_credito_maximo,
                c.dias_credito,
                c.saldo_credito_usado,
                c.fecha_creacion,
                c.fecha_actualizacion
            FROM clientes c
            WHERE c.activo = 1
            ORDER BY c.nombre ASC";

            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error en consulta: " . $conn->error);
                break;
            }

            $clientes = [];
            while ($row = $result->fetch_assoc()) {
                $cliente_id = intval($row['id']);

                // Obtener teléfonos del cliente
                $telefonosQuery = "SELECT telefono, tipo, whatsapp FROM cliente_telefonos WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $telefonosStmt = $conn->prepare($telefonosQuery);
                $telefonosStmt->bind_param("i", $cliente_id);
                $telefonosStmt->execute();
                $telefonosResult = $telefonosStmt->get_result();

                $telefonos = [];
                while ($telRow = $telefonosResult->fetch_assoc()) {
                    $telefonos[] = [
                        'telefono' => $telRow['telefono'],
                        'tipo' => $telRow['tipo'],
                        'whatsapp' => (bool)$telRow['whatsapp']
                    ];
                }

                // Obtener direcciones del cliente
                $direccionesQuery = "SELECT direccion, tipo, latitud, longitud FROM cliente_direcciones WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $direccionesStmt = $conn->prepare($direccionesQuery);
                $direccionesStmt->bind_param("i", $cliente_id);
                $direccionesStmt->execute();
                $direccionesResult = $direccionesStmt->get_result();

                $direcciones = [];
                while ($dirRow = $direccionesResult->fetch_assoc()) {
                    $direcciones[] = [
                        'direccion' => $dirRow['direccion'],
                        'es_principal' => $dirRow['tipo'] === 'principal',
                        'latitud' => $dirRow['latitud'] ? floatval($dirRow['latitud']) : null,
                        'longitud' => $dirRow['longitud'] ? floatval($dirRow['longitud']) : null
                    ];
                }

                // Obtener datos fiscales del cliente (si la tabla existe)
                $datos_fiscales = [];
                $datosFiscalesQuery = "SELECT id, rfc, razon_social, regimen_fiscal, uso_cfdi, codigo_postal_fiscal, email_facturacion, es_principal
                                       FROM cliente_datos_fiscales
                                       WHERE cliente_id = ? AND activo = 1
                                       ORDER BY es_principal DESC, id ASC";
                $datosFiscalesStmt = $conn->prepare($datosFiscalesQuery);
                if ($datosFiscalesStmt) {
                    $datosFiscalesStmt->bind_param("i", $cliente_id);
                    $datosFiscalesStmt->execute();
                    $datosFiscalesResult = $datosFiscalesStmt->get_result();

                    while ($dfRow = $datosFiscalesResult->fetch_assoc()) {
                        $datos_fiscales[] = [
                            'id' => intval($dfRow['id']),
                            'rfc' => $dfRow['rfc'],
                            'razon_social' => $dfRow['razon_social'],
                            'regimen_fiscal' => $dfRow['regimen_fiscal'],
                            'uso_cfdi' => $dfRow['uso_cfdi'] ?? '',
                            'codigo_postal_fiscal' => $dfRow['codigo_postal_fiscal'],
                            'email_facturacion' => $dfRow['email_facturacion'] ?? '',
                            'es_principal' => (bool)$dfRow['es_principal']
                        ];
                    }
                }

                $clientes[] = [
                    'id' => $cliente_id,
                    'nombre' => $row['nombre'],
                    'apellido' => '', // Para compatibilidad con frontend
                    'nombre_comercial' => $row['nombre_comercial'] ?? '',
                    'precio_venta' => $row['precio_venta'] ?? 'precio1',
                    'activo' => intval($row['activo']),
                    'tiene_credito' => intval($row['tiene_credito'] ?? 0),
                    'monto_credito_maximo' => floatval($row['monto_credito_maximo'] ?? 0),
                    'dias_credito' => intval($row['dias_credito'] ?? 0),
                    'saldo_credito_usado' => floatval($row['saldo_credito_usado'] ?? 0),
                    'fecha_creacion' => $row['fecha_creacion'],
                    'fecha_actualizacion' => $row['fecha_actualizacion'],
                    'telefonos' => $telefonos,
                    'direcciones' => $direcciones,
                    'datos_fiscales' => $datos_fiscales
                ];
            }

            sendResponse(true, "Clientes obtenidos exitosamente", $clientes);

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener clientes: " . $e->getMessage());
        }
        break;

    case 'getCliente':
        $id = intval($datapost['id'] ?? 0);

        if ($id <= 0) {
            sendResponse(false, "ID de cliente requerido");
        }

        try {
            // Obtener información básica del cliente
            $query = "SELECT
                id,
                nombre,
                nombre_comercial,
                precio_venta,
                activo,
                tiene_credito,
                monto_credito_maximo,
                dias_credito,
                saldo_credito_usado,
                fecha_creacion,
                fecha_actualizacion
            FROM clientes
            WHERE id = ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                $cliente_id = intval($row['id']);

                // Obtener teléfonos
                $telefonosQuery = "SELECT telefono, tipo, whatsapp FROM cliente_telefonos WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $telefonosStmt = $conn->prepare($telefonosQuery);
                $telefonosStmt->bind_param("i", $cliente_id);
                $telefonosStmt->execute();
                $telefonosResult = $telefonosStmt->get_result();

                $telefonos = [];
                while ($telRow = $telefonosResult->fetch_assoc()) {
                    $telefonos[] = [
                        'telefono' => $telRow['telefono'],
                        'tipo' => $telRow['tipo'],
                        'whatsapp' => (bool)$telRow['whatsapp']
                    ];
                }

                // Obtener direcciones
                $direccionesQuery = "SELECT direccion, tipo, latitud, longitud FROM cliente_direcciones WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $direccionesStmt = $conn->prepare($direccionesQuery);
                $direccionesStmt->bind_param("i", $cliente_id);
                $direccionesStmt->execute();
                $direccionesResult = $direccionesStmt->get_result();

                $direcciones = [];
                while ($dirRow = $direccionesResult->fetch_assoc()) {
                    $direcciones[] = [
                        'direccion' => $dirRow['direccion'],
                        'es_principal' => $dirRow['tipo'] === 'principal',
                        'latitud' => $dirRow['latitud'] ? floatval($dirRow['latitud']) : null,
                        'longitud' => $dirRow['longitud'] ? floatval($dirRow['longitud']) : null
                    ];
                }

                // Obtener datos fiscales (si la tabla existe)
                $datos_fiscales = [];
                $datosFiscalesQuery = "SELECT id, rfc, razon_social, regimen_fiscal, uso_cfdi, codigo_postal_fiscal, email_facturacion, es_principal
                                       FROM cliente_datos_fiscales
                                       WHERE cliente_id = ? AND activo = 1
                                       ORDER BY es_principal DESC, id ASC";
                $datosFiscalesStmt = $conn->prepare($datosFiscalesQuery);
                if ($datosFiscalesStmt) {
                    $datosFiscalesStmt->bind_param("i", $cliente_id);
                    $datosFiscalesStmt->execute();
                    $datosFiscalesResult = $datosFiscalesStmt->get_result();

                    while ($dfRow = $datosFiscalesResult->fetch_assoc()) {
                        $datos_fiscales[] = [
                            'id' => intval($dfRow['id']),
                            'rfc' => $dfRow['rfc'],
                            'razon_social' => $dfRow['razon_social'],
                            'regimen_fiscal' => $dfRow['regimen_fiscal'],
                            'uso_cfdi' => $dfRow['uso_cfdi'] ?? '',
                            'codigo_postal_fiscal' => $dfRow['codigo_postal_fiscal'],
                            'email_facturacion' => $dfRow['email_facturacion'] ?? '',
                            'es_principal' => (bool)$dfRow['es_principal']
                        ];
                    }
                }

                $cliente = [
                    'id' => $cliente_id,
                    'nombre' => $row['nombre'],
                    'apellido' => '', // Para compatibilidad
                    'nombre_comercial' => $row['nombre_comercial'] ?? '',
                    'precio_venta' => $row['precio_venta'] ?? 'precio1',
                    'activo' => intval($row['activo']),
                    'tiene_credito' => intval($row['tiene_credito'] ?? 0),
                    'monto_credito_maximo' => floatval($row['monto_credito_maximo'] ?? 0),
                    'dias_credito' => intval($row['dias_credito'] ?? 0),
                    'saldo_credito_usado' => floatval($row['saldo_credito_usado'] ?? 0),
                    'fecha_creacion' => $row['fecha_creacion'],
                    'fecha_actualizacion' => $row['fecha_actualizacion'],
                    'telefonos' => $telefonos,
                    'direcciones' => $direcciones,
                    'datos_fiscales' => $datos_fiscales
                ];

                sendResponse(true, "Cliente obtenido exitosamente", $cliente);
            } else {
                sendResponse(false, "Cliente no encontrado");
            }

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener cliente: " . $e->getMessage());
        }
        break;

    case 'crearCliente':
        $nombre = trim($datapost['nombre'] ?? '');
        $nombre_comercial = trim($datapost['nombre_comercial'] ?? '');
        $precio_venta = trim($datapost['precio_venta'] ?? 'precio1');
        $activo = isset($datapost['activo']) ? intval($datapost['activo']) : 1;
        $tiene_credito = isset($datapost['tiene_credito']) ? intval($datapost['tiene_credito']) : 0;
        $monto_credito_maximo = isset($datapost['monto_credito_maximo']) ? floatval($datapost['monto_credito_maximo']) : 0;
        $dias_credito = isset($datapost['dias_credito']) ? intval($datapost['dias_credito']) : 0;
        $telefonos = $datapost['telefonos'] ?? [];
        $direcciones = $datapost['direcciones'] ?? [];
        $datos_fiscales = $datapost['datos_fiscales'] ?? [];

        // Validaciones
        if (empty($nombre)) {
            sendResponse(false, "El nombre es requerido");
        }

        if (empty($telefonos) || !is_array($telefonos)) {
            sendResponse(false, "Debe proporcionar al menos un teléfono");
        }

        // Validar teléfonos
        foreach ($telefonos as $tel) {
            if (empty($tel['telefono']) || !preg_match('/^\d{10}$/', trim($tel['telefono']))) {
                sendResponse(false, "Todos los teléfonos deben tener 10 dígitos");
            }
        }

        try {
            // Iniciar transacción
            $conn->begin_transaction();

            // Verificar que no existan teléfonos duplicados
            foreach ($telefonos as $tel) {
                $telefono = trim($tel['telefono']);
                $checkQuery = "SELECT cliente_id FROM cliente_telefonos WHERE telefono = ? AND activo = 1";
                $checkStmt = $conn->prepare($checkQuery);
                $checkStmt->bind_param("s", $telefono);
                $checkStmt->execute();
                $checkResult = $checkStmt->get_result();

                if ($checkResult->num_rows > 0) {
                    $conn->rollback();
                    sendResponse(false, "Ya existe un cliente con el teléfono: " . $telefono);
                }
            }

            // Crear el cliente
            $query = "INSERT INTO clientes (nombre, nombre_comercial, precio_venta, activo, tiene_credito, monto_credito_maximo, dias_credito) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("sssiidi", $nombre, $nombre_comercial, $precio_venta, $activo, $tiene_credito, $monto_credito_maximo, $dias_credito);

            if (!$stmt->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al crear el cliente");
            }

            $cliente_id = $conn->insert_id;

            // Insertar teléfonos
            $telefonoQuery = "INSERT INTO cliente_telefonos (cliente_id, telefono, tipo, whatsapp, activo) VALUES (?, ?, ?, ?, 1)";
            $telefonoStmt = $conn->prepare($telefonoQuery);

            foreach ($telefonos as $tel) {
                $telefono = trim($tel['telefono']);
                $tipo = $tel['tipo'] ?? 'principal';
                $whatsapp = isset($tel['whatsapp']) ? intval($tel['whatsapp']) : 0;
                $telefonoStmt->bind_param("issi", $cliente_id, $telefono, $tipo, $whatsapp);

                if (!$telefonoStmt->execute()) {
                    $conn->rollback();
                    sendResponse(false, "Error al guardar teléfono: " . $telefono);
                }
            }

            // Insertar direcciones (si hay)
            if (!empty($direcciones) && is_array($direcciones)) {
                $direccionQuery = "INSERT INTO cliente_direcciones (cliente_id, direccion, tipo, latitud, longitud, activo) VALUES (?, ?, ?, ?, ?, 1)";
                $direccionStmt = $conn->prepare($direccionQuery);

                foreach ($direcciones as $dir) {
                    $direccion = trim($dir['direccion']);
                    if (!empty($direccion)) {
                        $tipo = ($dir['es_principal'] ?? false) ? 'principal' : 'otro';
                        $latitud = isset($dir['latitud']) && $dir['latitud'] !== null ? floatval($dir['latitud']) : null;
                        $longitud = isset($dir['longitud']) && $dir['longitud'] !== null ? floatval($dir['longitud']) : null;
                        $direccionStmt->bind_param("issdd", $cliente_id, $direccion, $tipo, $latitud, $longitud);

                        if (!$direccionStmt->execute()) {
                            $conn->rollback();
                            sendResponse(false, "Error al guardar dirección");
                        }
                    }
                }
            }

            // Insertar datos fiscales (si hay y la tabla existe)
            if (!empty($datos_fiscales) && is_array($datos_fiscales)) {
                $datoFiscalQuery = "INSERT INTO cliente_datos_fiscales (cliente_id, rfc, razon_social, regimen_fiscal, uso_cfdi, codigo_postal_fiscal, email_facturacion, es_principal, activo) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1)";
                $datoFiscalStmt = $conn->prepare($datoFiscalQuery);

                if ($datoFiscalStmt) {
                    foreach ($datos_fiscales as $df) {
                        $rfc = strtoupper(trim($df['rfc'] ?? ''));
                        if (!empty($rfc)) {
                            $razon_social = trim($df['razon_social'] ?? '');
                            $regimen_fiscal = trim($df['regimen_fiscal'] ?? '');
                            $uso_cfdi = trim($df['uso_cfdi'] ?? '');
                            $codigo_postal_fiscal = trim($df['codigo_postal_fiscal'] ?? '');
                            $email_facturacion = trim($df['email_facturacion'] ?? '');
                            $es_principal = ($df['es_principal'] ?? false) ? 1 : 0;

                            $datoFiscalStmt->bind_param("issssssi", $cliente_id, $rfc, $razon_social, $regimen_fiscal, $uso_cfdi, $codigo_postal_fiscal, $email_facturacion, $es_principal);

                            if (!$datoFiscalStmt->execute()) {
                                $conn->rollback();
                                sendResponse(false, "Error al guardar datos fiscales");
                            }
                        }
                    }
                }
            }

            // Confirmar transacción
            $conn->commit();
            sendResponse(true, "Cliente creado exitosamente", ['id' => $cliente_id]);

        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al crear cliente: " . $e->getMessage());
        }
        break;

    case 'actualizarCliente':
        $id = intval($datapost['id'] ?? 0);
        $nombre = trim($datapost['nombre'] ?? '');
        $nombre_comercial = trim($datapost['nombre_comercial'] ?? '');
        $precio_venta = trim($datapost['precio_venta'] ?? 'precio1');
        $activo = isset($datapost['activo']) ? intval($datapost['activo']) : 1;
        $tiene_credito = isset($datapost['tiene_credito']) ? intval($datapost['tiene_credito']) : 0;
        $monto_credito_maximo = isset($datapost['monto_credito_maximo']) ? floatval($datapost['monto_credito_maximo']) : 0;
        $dias_credito = isset($datapost['dias_credito']) ? intval($datapost['dias_credito']) : 0;
        $telefonos = $datapost['telefonos'] ?? [];
        $direcciones = $datapost['direcciones'] ?? [];
        $datos_fiscales = $datapost['datos_fiscales'] ?? [];

        if ($id <= 0) {
            sendResponse(false, "ID de cliente requerido");
        }

        // Validaciones
        if (empty($nombre)) {
            sendResponse(false, "El nombre es requerido");
        }

        if (empty($telefonos) || !is_array($telefonos)) {
            sendResponse(false, "Debe proporcionar al menos un teléfono");
        }

        // Validar teléfonos
        foreach ($telefonos as $tel) {
            if (empty($tel['telefono']) || !preg_match('/^\d{10}$/', trim($tel['telefono']))) {
                sendResponse(false, "Todos los teléfonos deben tener 10 dígitos");
            }
        }

        try {
            // Verificar que el cliente existe
            $checkQuery = "SELECT nombre FROM clientes WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows === 0) {
                sendResponse(false, "Cliente no encontrado");
            }

            // Iniciar transacción
            $conn->begin_transaction();

            // Verificar que no existan teléfonos duplicados en otros clientes
            foreach ($telefonos as $tel) {
                $telefono = trim($tel['telefono']);
                $checkTelQuery = "SELECT cliente_id FROM cliente_telefonos WHERE telefono = ? AND cliente_id != ? AND activo = 1";
                $checkTelStmt = $conn->prepare($checkTelQuery);
                $checkTelStmt->bind_param("si", $telefono, $id);
                $checkTelStmt->execute();
                $checkTelResult = $checkTelStmt->get_result();

                if ($checkTelResult->num_rows > 0) {
                    $conn->rollback();
                    sendResponse(false, "Ya existe otro cliente con el teléfono: " . $telefono);
                }
            }

            // Actualizar información básica del cliente
            $updateQuery = "UPDATE clientes SET nombre = ?, nombre_comercial = ?, precio_venta = ?, activo = ?, tiene_credito = ?, monto_credito_maximo = ?, dias_credito = ? WHERE id = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->bind_param("sssiidii", $nombre, $nombre_comercial, $precio_venta, $activo, $tiene_credito, $monto_credito_maximo, $dias_credito, $id);

            if (!$updateStmt->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al actualizar el cliente");
            }

            // Marcar todos los teléfonos como inactivos primero
            $deleteTelQuery = "UPDATE cliente_telefonos SET activo = 0 WHERE cliente_id = ?";
            $deleteTelStmt = $conn->prepare($deleteTelQuery);
            $deleteTelStmt->bind_param("i", $id);
            $deleteTelStmt->execute();

            // Insertar o actualizar teléfonos
            foreach ($telefonos as $tel) {
                $telefono = trim($tel['telefono']);
                $tipo = $tel['tipo'] ?? 'principal';
                $whatsapp = isset($tel['whatsapp']) ? intval($tel['whatsapp']) : 0;

                // Verificar si el teléfono ya existe para este cliente (activo o inactivo)
                $checkExistingQuery = "SELECT id FROM cliente_telefonos WHERE cliente_id = ? AND telefono = ?";
                $checkExistingStmt = $conn->prepare($checkExistingQuery);
                $checkExistingStmt->bind_param("is", $id, $telefono);
                $checkExistingStmt->execute();
                $checkExistingResult = $checkExistingStmt->get_result();

                if ($checkExistingResult->num_rows > 0) {
                    // El teléfono ya existe, solo reactivarlo y actualizar tipo y whatsapp
                    $updateExistingQuery = "UPDATE cliente_telefonos SET tipo = ?, whatsapp = ?, activo = 1 WHERE cliente_id = ? AND telefono = ?";
                    $updateExistingStmt = $conn->prepare($updateExistingQuery);
                    $updateExistingStmt->bind_param("siis", $tipo, $whatsapp, $id, $telefono);

                    if (!$updateExistingStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al actualizar teléfono: " . $telefono);
                    }
                } else {
                    // El teléfono no existe, insertarlo
                    $insertTelQuery = "INSERT INTO cliente_telefonos (cliente_id, telefono, tipo, whatsapp, activo) VALUES (?, ?, ?, ?, 1)";
                    $insertTelStmt = $conn->prepare($insertTelQuery);
                    $insertTelStmt->bind_param("issi", $id, $telefono, $tipo, $whatsapp);

                    if (!$insertTelStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al insertar teléfono: " . $telefono);
                    }
                }
            }

            // Marcar todas las direcciones como inactivas primero
            $deleteDirQuery = "UPDATE cliente_direcciones SET activo = 0 WHERE cliente_id = ?";
            $deleteDirStmt = $conn->prepare($deleteDirQuery);
            $deleteDirStmt->bind_param("i", $id);
            $deleteDirStmt->execute();

            // Insertar o actualizar direcciones (si hay)
            if (!empty($direcciones) && is_array($direcciones)) {
                foreach ($direcciones as $dir) {
                    $direccion = trim($dir['direccion']);
                    if (!empty($direccion)) {
                        $tipo = ($dir['es_principal'] ?? false) ? 'principal' : 'otro';
                        $latitud = isset($dir['latitud']) && $dir['latitud'] !== null ? floatval($dir['latitud']) : null;
                        $longitud = isset($dir['longitud']) && $dir['longitud'] !== null ? floatval($dir['longitud']) : null;

                        // Verificar si la dirección ya existe para este cliente
                        $checkExistingDirQuery = "SELECT id FROM cliente_direcciones WHERE cliente_id = ? AND direccion = ?";
                        $checkExistingDirStmt = $conn->prepare($checkExistingDirQuery);
                        $checkExistingDirStmt->bind_param("is", $id, $direccion);
                        $checkExistingDirStmt->execute();
                        $checkExistingDirResult = $checkExistingDirStmt->get_result();

                        if ($checkExistingDirResult->num_rows > 0) {
                            // La dirección ya existe, solo reactivarla y actualizar
                            $updateExistingDirQuery = "UPDATE cliente_direcciones SET tipo = ?, latitud = ?, longitud = ?, activo = 1 WHERE cliente_id = ? AND direccion = ?";
                            $updateExistingDirStmt = $conn->prepare($updateExistingDirQuery);
                            $updateExistingDirStmt->bind_param("sddis", $tipo, $latitud, $longitud, $id, $direccion);

                            if (!$updateExistingDirStmt->execute()) {
                                $conn->rollback();
                                sendResponse(false, "Error al actualizar dirección");
                            }
                        } else {
                            // La dirección no existe, insertarla
                            $insertDirQuery = "INSERT INTO cliente_direcciones (cliente_id, direccion, tipo, latitud, longitud, activo) VALUES (?, ?, ?, ?, ?, 1)";
                            $insertDirStmt = $conn->prepare($insertDirQuery);
                            $insertDirStmt->bind_param("issdd", $id, $direccion, $tipo, $latitud, $longitud);

                            if (!$insertDirStmt->execute()) {
                                $conn->rollback();
                                sendResponse(false, "Error al insertar dirección");
                            }
                        }
                    }
                }
            }

            // Marcar todos los datos fiscales como inactivos primero (si la tabla existe)
            $deleteDfQuery = "UPDATE cliente_datos_fiscales SET activo = 0 WHERE cliente_id = ?";
            $deleteDfStmt = $conn->prepare($deleteDfQuery);
            if ($deleteDfStmt) {
                $deleteDfStmt->bind_param("i", $id);
                $deleteDfStmt->execute();
            }

            // Insertar o actualizar datos fiscales (si hay y la tabla existe)
            if (!empty($datos_fiscales) && is_array($datos_fiscales)) {
                // Verificar si la tabla existe
                $checkExistingDfQuery = "SELECT id FROM cliente_datos_fiscales WHERE cliente_id = ? AND rfc = ? LIMIT 1";
                $checkExistingDfStmt = $conn->prepare($checkExistingDfQuery);

                if ($checkExistingDfStmt) {
                    foreach ($datos_fiscales as $df) {
                        $rfc = strtoupper(trim($df['rfc'] ?? ''));
                        if (!empty($rfc)) {
                            $razon_social = trim($df['razon_social'] ?? '');
                            $regimen_fiscal = trim($df['regimen_fiscal'] ?? '');
                            $uso_cfdi = trim($df['uso_cfdi'] ?? '');
                            $codigo_postal_fiscal = trim($df['codigo_postal_fiscal'] ?? '');
                            $email_facturacion = trim($df['email_facturacion'] ?? '');
                            $es_principal = ($df['es_principal'] ?? false) ? 1 : 0;

                            // Verificar si el RFC ya existe para este cliente
                            $checkExistingDfStmt->bind_param("is", $id, $rfc);
                            $checkExistingDfStmt->execute();
                            $checkExistingDfResult = $checkExistingDfStmt->get_result();

                            if ($checkExistingDfResult->num_rows > 0) {
                                // El dato fiscal ya existe, actualizarlo
                                $updateExistingDfQuery = "UPDATE cliente_datos_fiscales
                                                          SET razon_social = ?, regimen_fiscal = ?, uso_cfdi = ?,
                                                              codigo_postal_fiscal = ?, email_facturacion = ?,
                                                              es_principal = ?, activo = 1
                                                          WHERE cliente_id = ? AND rfc = ?";
                                $updateExistingDfStmt = $conn->prepare($updateExistingDfQuery);
                                if ($updateExistingDfStmt) {
                                    $updateExistingDfStmt->bind_param("sssssiis", $razon_social, $regimen_fiscal, $uso_cfdi, $codigo_postal_fiscal, $email_facturacion, $es_principal, $id, $rfc);

                                    if (!$updateExistingDfStmt->execute()) {
                                        $conn->rollback();
                                        sendResponse(false, "Error al actualizar datos fiscales");
                                    }
                                }
                            } else {
                                // El dato fiscal no existe, insertarlo
                                $insertDfQuery = "INSERT INTO cliente_datos_fiscales (cliente_id, rfc, razon_social, regimen_fiscal, uso_cfdi, codigo_postal_fiscal, email_facturacion, es_principal, activo)
                                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1)";
                                $insertDfStmt = $conn->prepare($insertDfQuery);
                                if ($insertDfStmt) {
                                    $insertDfStmt->bind_param("issssssi", $id, $rfc, $razon_social, $regimen_fiscal, $uso_cfdi, $codigo_postal_fiscal, $email_facturacion, $es_principal);

                                    if (!$insertDfStmt->execute()) {
                                        $conn->rollback();
                                        sendResponse(false, "Error al insertar datos fiscales");
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // Confirmar transacción
            $conn->commit();
            sendResponse(true, "Cliente actualizado exitosamente");

        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al actualizar cliente: " . $e->getMessage());
        }
        break;

    case 'eliminarCliente':
        $id = intval($datapost['id'] ?? 0);

        if ($id <= 0) {
            sendResponse(false, "ID de cliente requerido");
        }

        try {
            // Verificar que el cliente existe
            $checkQuery = "SELECT nombre FROM clientes WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows === 0) {
                sendResponse(false, "Cliente no encontrado");
            }

            // Iniciar transacción
            $conn->begin_transaction();

            // Marcar cliente como inactivo (soft delete)
            $updateClienteQuery = "UPDATE clientes SET activo = 0 WHERE id = ?";
            $updateClienteStmt = $conn->prepare($updateClienteQuery);
            $updateClienteStmt->bind_param("i", $id);

            // Marcar teléfonos como inactivos
            $updateTelefonosQuery = "UPDATE cliente_telefonos SET activo = 0 WHERE cliente_id = ?";
            $updateTelefonosStmt = $conn->prepare($updateTelefonosQuery);
            $updateTelefonosStmt->bind_param("i", $id);

            // Marcar direcciones como inactivas
            $updateDireccionesQuery = "UPDATE cliente_direcciones SET activo = 0 WHERE cliente_id = ?";
            $updateDireccionesStmt = $conn->prepare($updateDireccionesQuery);
            $updateDireccionesStmt->bind_param("i", $id);

            // Marcar datos fiscales como inactivos (si la tabla existe)
            $updateDatosFiscalesQuery = "UPDATE cliente_datos_fiscales SET activo = 0 WHERE cliente_id = ?";
            $updateDatosFiscalesStmt = $conn->prepare($updateDatosFiscalesQuery);
            if ($updateDatosFiscalesStmt) {
                $updateDatosFiscalesStmt->bind_param("i", $id);
                $updateDatosFiscalesStmt->execute();
            }

            if ($updateClienteStmt->execute() && $updateTelefonosStmt->execute() && $updateDireccionesStmt->execute()) {
                $conn->commit();
                sendResponse(true, "Cliente desactivado exitosamente");
            } else {
                $conn->rollback();
                sendResponse(false, "Error al desactivar el cliente");
            }

        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al eliminar cliente: " . $e->getMessage());
        }
        break;

    case 'buscarClientes':
        $query_search = trim($datapost['query'] ?? '');

        if (empty($query_search)) {
            sendResponse(false, "Término de búsqueda requerido");
        }

        try {
            $query = "SELECT DISTINCT c.id, c.nombre, c.nombre_comercial, c.precio_venta, c.activo, c.tiene_credito, c.monto_credito_maximo, c.dias_credito, c.saldo_credito_usado
                     FROM clientes c
                     LEFT JOIN cliente_telefonos ct ON c.id = ct.cliente_id AND ct.activo = 1
                     LEFT JOIN cliente_direcciones cd ON c.id = cd.cliente_id AND cd.activo = 1
                     WHERE c.activo = 1 AND (
                         c.nombre LIKE ? OR
                         c.nombre_comercial LIKE ? OR
                         ct.telefono LIKE ? OR
                         cd.direccion LIKE ?
                     )
                     ORDER BY c.nombre ASC
                     LIMIT 50";

            $searchTerm = "%{$query_search}%";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("ssss", $searchTerm, $searchTerm, $searchTerm, $searchTerm);
            $stmt->execute();
            $result = $stmt->get_result();

            $clientes = [];
            while ($row = $result->fetch_assoc()) {
                $cliente_id = intval($row['id']);

                // Obtener todos los teléfonos del cliente
                $telefonosQuery = "SELECT telefono, tipo, whatsapp FROM cliente_telefonos WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $telefonosStmt = $conn->prepare($telefonosQuery);
                $telefonosStmt->bind_param("i", $cliente_id);
                $telefonosStmt->execute();
                $telefonosResult = $telefonosStmt->get_result();

                $telefonos = [];
                while ($telRow = $telefonosResult->fetch_assoc()) {
                    $telefonos[] = [
                        'telefono' => $telRow['telefono'],
                        'tipo' => $telRow['tipo'],
                        'whatsapp' => (bool)$telRow['whatsapp']
                    ];
                }

                // Obtener todas las direcciones del cliente
                $direccionesQuery = "SELECT direccion, tipo, latitud, longitud FROM cliente_direcciones WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $direccionesStmt = $conn->prepare($direccionesQuery);
                $direccionesStmt->bind_param("i", $cliente_id);
                $direccionesStmt->execute();
                $direccionesResult = $direccionesStmt->get_result();

                $direcciones = [];
                while ($dirRow = $direccionesResult->fetch_assoc()) {
                    $direcciones[] = [
                        'direccion' => $dirRow['direccion'],
                        'es_principal' => $dirRow['tipo'] === 'principal',
                        'latitud' => $dirRow['latitud'] ? floatval($dirRow['latitud']) : null,
                        'longitud' => $dirRow['longitud'] ? floatval($dirRow['longitud']) : null
                    ];
                }

                $clientes[] = [
                    'id' => $cliente_id,
                    'nombre' => $row['nombre'],
                    'apellido' => '',
                    'nombre_comercial' => $row['nombre_comercial'] ?? '',
                    'precio_venta' => $row['precio_venta'] ?? 'precio1',
                    'activo' => intval($row['activo']),
                    'tiene_credito' => intval($row['tiene_credito'] ?? 0),
                    'monto_credito_maximo' => floatval($row['monto_credito_maximo'] ?? 0),
                    'dias_credito' => intval($row['dias_credito'] ?? 0),
                    'saldo_credito_usado' => floatval($row['saldo_credito_usado'] ?? 0),
                    'telefonos' => $telefonos,
                    'direcciones' => $direcciones
                ];
            }

            sendResponse(true, "Búsqueda completada", $clientes);

        } catch (Exception $e) {
            sendResponse(false, "Error al buscar clientes: " . $e->getMessage());
        }
        break;

    case 'buscarClientePorTelefono':
        $telefono = trim($datapost['telefono'] ?? '');

        if (empty($telefono)) {
            sendResponse(false, "Número de teléfono requerido");
        }

        try {
            $query = "SELECT c.id, c.nombre, c.nombre_comercial, c.precio_venta, c.activo, c.tiene_credito, c.monto_credito_maximo, c.dias_credito, c.saldo_credito_usado
                     FROM clientes c
                     JOIN cliente_telefonos ct ON c.id = ct.cliente_id
                     WHERE ct.telefono = ? AND c.activo = 1 AND ct.activo = 1
                     LIMIT 1";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("s", $telefono);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                $cliente_id = intval($row['id']);

                // Obtener todos los teléfonos del cliente
                $telefonosQuery = "SELECT telefono, tipo, whatsapp FROM cliente_telefonos WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $telefonosStmt = $conn->prepare($telefonosQuery);
                $telefonosStmt->bind_param("i", $cliente_id);
                $telefonosStmt->execute();
                $telefonosResult = $telefonosStmt->get_result();

                $telefonos = [];
                while ($telRow = $telefonosResult->fetch_assoc()) {
                    $telefonos[] = [
                        'telefono' => $telRow['telefono'],
                        'tipo' => $telRow['tipo'],
                        'whatsapp' => (bool)$telRow['whatsapp']
                    ];
                }

                // Obtener todas las direcciones del cliente
                $direccionesQuery = "SELECT direccion, tipo, latitud, longitud FROM cliente_direcciones WHERE cliente_id = ? AND activo = 1 ORDER BY id ASC";
                $direccionesStmt = $conn->prepare($direccionesQuery);
                $direccionesStmt->bind_param("i", $cliente_id);
                $direccionesStmt->execute();
                $direccionesResult = $direccionesStmt->get_result();

                $direcciones = [];
                while ($dirRow = $direccionesResult->fetch_assoc()) {
                    $direcciones[] = [
                        'direccion' => $dirRow['direccion'],
                        'es_principal' => $dirRow['tipo'] === 'principal',
                        'latitud' => $dirRow['latitud'] ? floatval($dirRow['latitud']) : null,
                        'longitud' => $dirRow['longitud'] ? floatval($dirRow['longitud']) : null
                    ];
                }

                $cliente = [
                    'id' => $cliente_id,
                    'nombre' => $row['nombre'],
                    'apellido' => '',
                    'nombre_comercial' => $row['nombre_comercial'] ?? '',
                    'precio_venta' => $row['precio_venta'] ?? 'precio1',
                    'telefono_buscado' => $telefono,
                    'activo' => intval($row['activo']),
                    'tiene_credito' => intval($row['tiene_credito'] ?? 0),
                    'monto_credito_maximo' => floatval($row['monto_credito_maximo'] ?? 0),
                    'dias_credito' => intval($row['dias_credito'] ?? 0),
                    'saldo_credito_usado' => floatval($row['saldo_credito_usado'] ?? 0),
                    'telefonos' => $telefonos,
                    'direcciones' => $direcciones
                ];

                sendResponse(true, "Cliente encontrado", $cliente);
            } else {
                sendResponse(false, "No se encontró un cliente con ese número de teléfono");
            }

        } catch (Exception $e) {
            sendResponse(false, "Error al buscar cliente: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para clientes");
        break;
}
?>