<?php
/**
 * Configuraciones.php
 * Endpoints para gestionar configuraciones del sistema
 * Soporta configuraciones globales y por caja/punto de venta
 * La caja se identifica por nombre_pc (hostname del equipo)
 */

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

switch ($tipo) {
    case 'getConfiguraciones':
        cfg_getConfiguraciones();
        break;
    case 'getConfiguracion':
        cfg_getConfiguracion();
        break;
    case 'getConfiguracionesPorCategoria':
        cfg_getConfiguracionesPorCategoria();
        break;
    case 'actualizarConfiguracion':
        cfg_actualizarConfiguracion();
        break;
    case 'actualizarConfiguraciones':
        cfg_actualizarConfiguraciones();
        break;
    case 'registrarCaja':
        cfg_registrarCaja();
        break;
    case 'getCajas':
        cfg_getCajas();
        break;
    case 'getCajaPorNombrePc':
        cfg_getCajaPorNombrePc();
        break;
    default:
        sendResponse(false, "Operación de configuraciones no implementada: " . $tipo);
        break;
}

/**
 * Obtener el ID de la caja por nombre_pc
 */
function cfg_obtenerCajaIdPorNombrePc($nombrePc) {
    global $conn;

    if (!$nombrePc) return null;

    $stmt = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
    $stmt->bind_param("s", $nombrePc);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($row = $result->fetch_assoc()) {
        return $row['id'];
    }

    return null;
}

/**
 * Obtener todas las configuraciones (globales o por caja)
 */
function cfg_getConfiguraciones() {
    global $conn, $datapost;

    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);
    $cajaId = cfg_obtenerCajaIdPorNombrePc($nombrePc);

    try {
        if ($cajaId) {
            // Obtener configs específicas de la caja, con fallback a globales
            $query = "
                SELECT
                    COALESCE(c_caja.id, c_global.id) as id,
                    COALESCE(c_caja.clave, c_global.clave) as clave,
                    COALESCE(c_caja.valor, c_global.valor) as valor,
                    COALESCE(c_caja.tipo, c_global.tipo) as tipo,
                    COALESCE(c_caja.categoria, c_global.categoria) as categoria,
                    COALESCE(c_caja.descripcion, c_global.descripcion) as descripcion,
                    c_caja.caja_id
                FROM configuraciones c_global
                LEFT JOIN configuraciones c_caja ON c_global.clave = c_caja.clave AND c_caja.caja_id = ?
                WHERE c_global.caja_id IS NULL
                ORDER BY categoria, clave
            ";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $cajaId);
            $stmt->execute();
            $result = $stmt->get_result();
        } else {
            // Solo configs globales
            $query = "SELECT id, clave, valor, tipo, categoria, descripcion, caja_id FROM configuraciones WHERE caja_id IS NULL ORDER BY categoria, clave";
            $result = $conn->query($query);
        }

        if (!$result) {
            throw new Exception("Error al obtener configuraciones: " . $conn->error);
        }

        $configuraciones = [];
        while ($row = $result->fetch_assoc()) {
            $row['valor'] = cfg_convertirValor($row['valor'], $row['tipo']);
            $configuraciones[] = $row;
        }

        sendResponse(true, "Configuraciones obtenidas", $configuraciones);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener una configuración por clave (con soporte para caja)
 */
function cfg_getConfiguracion() {
    global $conn, $datapost;

    $clave = isset($_GET['clave']) ? $_GET['clave'] : (isset($datapost['clave']) ? $datapost['clave'] : null);
    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);
    $cajaId = cfg_obtenerCajaIdPorNombrePc($nombrePc);

    if (!$clave) {
        sendResponse(false, "Clave de configuración requerida");
        return;
    }

    try {
        if ($cajaId) {
            // Primero buscar config específica de la caja, luego global
            $stmt = $conn->prepare("
                SELECT id, clave, valor, tipo, categoria, descripcion, caja_id
                FROM configuraciones
                WHERE clave = ? AND (caja_id = ? OR caja_id IS NULL)
                ORDER BY caja_id DESC
                LIMIT 1
            ");
            $stmt->bind_param("si", $clave, $cajaId);
        } else {
            $stmt = $conn->prepare("SELECT id, clave, valor, tipo, categoria, descripcion, caja_id FROM configuraciones WHERE clave = ? AND caja_id IS NULL");
            $stmt->bind_param("s", $clave);
        }

        $stmt->execute();
        $result = $stmt->get_result();
        $config = $result->fetch_assoc();

        if (!$config) {
            sendResponse(false, "Configuración no encontrada");
            return;
        }

        $config['valor'] = cfg_convertirValor($config['valor'], $config['tipo']);

        sendResponse(true, "Configuración obtenida", $config);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener configuraciones por categoría (con soporte para caja)
 */
function cfg_getConfiguracionesPorCategoria() {
    global $conn, $datapost;

    $categoria = isset($_GET['categoria']) ? $_GET['categoria'] : (isset($datapost['categoria']) ? $datapost['categoria'] : null);
    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);
    $cajaId = cfg_obtenerCajaIdPorNombrePc($nombrePc);

    if (!$categoria) {
        sendResponse(false, "Categoría requerida");
        return;
    }

    try {
        if ($cajaId) {
            // Obtener configs específicas de la caja con fallback a globales
            // Incluye configs que solo existen para la caja (sin global correspondiente)
            $query = "
                SELECT
                    COALESCE(c_caja.id, c_global.id) as id,
                    COALESCE(c_caja.clave, c_global.clave) as clave,
                    COALESCE(c_caja.valor, c_global.valor) as valor,
                    COALESCE(c_caja.tipo, c_global.tipo) as tipo,
                    COALESCE(c_caja.categoria, c_global.categoria) as categoria,
                    COALESCE(c_caja.descripcion, c_global.descripcion) as descripcion,
                    c_caja.caja_id
                FROM configuraciones c_global
                LEFT JOIN configuraciones c_caja ON c_global.clave = c_caja.clave AND c_caja.caja_id = ?
                WHERE c_global.caja_id IS NULL AND c_global.categoria = ?

                UNION

                SELECT
                    c_solo_caja.id,
                    c_solo_caja.clave,
                    c_solo_caja.valor,
                    c_solo_caja.tipo,
                    c_solo_caja.categoria,
                    c_solo_caja.descripcion,
                    c_solo_caja.caja_id
                FROM configuraciones c_solo_caja
                WHERE c_solo_caja.caja_id = ? AND c_solo_caja.categoria = ?
                AND NOT EXISTS (
                    SELECT 1 FROM configuraciones c_g
                    WHERE c_g.clave = c_solo_caja.clave AND c_g.caja_id IS NULL
                )

                ORDER BY clave
            ";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("isis", $cajaId, $categoria, $cajaId, $categoria);
            $stmt->execute();
            $result = $stmt->get_result();
        } else {
            $stmt = $conn->prepare("SELECT id, clave, valor, tipo, categoria, descripcion, caja_id FROM configuraciones WHERE categoria = ? AND caja_id IS NULL ORDER BY clave");
            $stmt->bind_param("s", $categoria);
            $stmt->execute();
            $result = $stmt->get_result();
        }

        $configuraciones = [];
        while ($row = $result->fetch_assoc()) {
            $row['valor'] = cfg_convertirValor($row['valor'], $row['tipo']);
            $configuraciones[] = $row;
        }

        // También devolver como objeto indexado por clave para fácil acceso
        $configuracionesMap = [];
        foreach ($configuraciones as $config) {
            $configuracionesMap[$config['clave']] = $config['valor'];
        }

        sendResponse(true, "Configuraciones obtenidas", [
            'lista' => $configuraciones,
            'valores' => $configuracionesMap,
            'caja_id' => $cajaId
        ]);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Actualizar una configuración (con soporte para caja)
 */
function cfg_actualizarConfiguracion() {
    global $conn, $datapost;

    $clave = isset($datapost['clave']) ? $datapost['clave'] : null;
    $valor = isset($datapost['valor']) ? $datapost['valor'] : null;
    $nombrePc = isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null;
    $cajaId = cfg_obtenerCajaIdPorNombrePc($nombrePc);

    if (!$clave) {
        sendResponse(false, "Clave de configuración requerida");
        return;
    }

    try {
        // Obtener tipo de la configuración global
        $stmt = $conn->prepare("SELECT tipo FROM configuraciones WHERE clave = ? AND caja_id IS NULL");
        $stmt->bind_param("s", $clave);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows == 0) {
            sendResponse(false, "Configuración no encontrada");
            return;
        }

        $configRow = $result->fetch_assoc();
        $tipo = $configRow['tipo'];
        $valorStr = cfg_convertirParaAlmacenar($valor, $tipo);

        if ($cajaId) {
            // Verificar si ya existe config para esta caja
            $stmt = $conn->prepare("SELECT id FROM configuraciones WHERE clave = ? AND caja_id = ?");
            $stmt->bind_param("si", $clave, $cajaId);
            $stmt->execute();
            $existeResult = $stmt->get_result();

            if ($existeResult->num_rows > 0) {
                // Actualizar existente
                $stmt = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id = ?");
                $stmt->bind_param("ssi", $valorStr, $clave, $cajaId);
            } else {
                // Insertar nuevo para esta caja
                $stmt = $conn->prepare("
                    INSERT INTO configuraciones (clave, valor, tipo, categoria, descripcion, caja_id, created_at, updated_at)
                    SELECT clave, ?, tipo, categoria, descripcion, ?, NOW(), NOW()
                    FROM configuraciones WHERE clave = ? AND caja_id IS NULL
                ");
                $stmt->bind_param("sis", $valorStr, $cajaId, $clave);
            }
        } else {
            // Actualizar config global
            $stmt = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id IS NULL");
            $stmt->bind_param("ss", $valorStr, $clave);
        }

        if ($stmt->execute()) {
            sendResponse(true, "Configuración actualizada correctamente");
        } else {
            throw new Exception("Error al actualizar configuración: " . $conn->error);
        }
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Actualizar múltiples configuraciones (con soporte para caja)
 */
function cfg_actualizarConfiguraciones() {
    global $conn, $datapost;

    $configuraciones = isset($datapost['configuraciones']) ? $datapost['configuraciones'] : [];
    $nombrePc = isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null;
    $cajaId = cfg_obtenerCajaIdPorNombrePc($nombrePc);

    if (empty($configuraciones)) {
        sendResponse(false, "No se proporcionaron configuraciones para actualizar");
        return;
    }

    try {
        $conn->begin_transaction();

        foreach ($configuraciones as $config) {
            if (!isset($config['clave'])) continue;

            $clave = $config['clave'];
            $valor = isset($config['valor']) ? $config['valor'] : '';

            // Obtener tipo de la configuración global
            $stmt = $conn->prepare("SELECT tipo, categoria, descripcion FROM configuraciones WHERE clave = ? AND caja_id IS NULL");
            $stmt->bind_param("s", $clave);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows == 0) {
                // La configuración global no existe
                // Para configuraciones de caja (caja_*), crear SOLO la configuración de caja (sin global)
                if (strpos($clave, 'caja_') === 0 && $cajaId) {
                    // Determinar tipo basado en el valor
                    $tipo = is_numeric($valor) ? 'number' : (is_bool($valor) || $valor === 'true' || $valor === 'false' ? 'boolean' : 'string');
                    $categoria = 'caja';
                    $descripcion = 'Configuración de caja: ' . str_replace('_', ' ', substr($clave, 5));
                    $valorStr = cfg_convertirParaAlmacenar($valor, $tipo);

                    // Verificar si ya existe para esta caja
                    $stmtCheck = $conn->prepare("SELECT id FROM configuraciones WHERE clave = ? AND caja_id = ?");
                    $stmtCheck->bind_param("si", $clave, $cajaId);
                    $stmtCheck->execute();
                    $checkResult = $stmtCheck->get_result();

                    if ($checkResult->num_rows > 0) {
                        // Actualizar existente
                        $stmtUpdate = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id = ?");
                        $stmtUpdate->bind_param("ssi", $valorStr, $clave, $cajaId);
                        $stmtUpdate->execute();
                    } else {
                        // Crear configuración SOLO para esta caja (sin global)
                        $stmtCaja = $conn->prepare("
                            INSERT INTO configuraciones (clave, valor, tipo, categoria, descripcion, caja_id, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())
                        ");
                        $stmtCaja->bind_param("sssssi", $clave, $valorStr, $tipo, $categoria, $descripcion, $cajaId);
                        $stmtCaja->execute();
                    }
                }
                // Para configuraciones de impresión (impresion_*), crear SOLO la configuración de caja (sin global)
                else if (strpos($clave, 'impresion_') === 0 && $cajaId) {
                    $tipo = is_numeric($valor) ? 'number' : (is_bool($valor) || $valor === 'true' || $valor === 'false' ? 'boolean' : 'string');
                    $categoria = 'impresion';
                    $descripcion = 'Configuración de impresión: ' . str_replace('_', ' ', substr($clave, 10));
                    $valorStr = cfg_convertirParaAlmacenar($valor, $tipo);

                    $stmtCheck = $conn->prepare("SELECT id FROM configuraciones WHERE clave = ? AND caja_id = ?");
                    $stmtCheck->bind_param("si", $clave, $cajaId);
                    $stmtCheck->execute();
                    $checkResult = $stmtCheck->get_result();

                    if ($checkResult->num_rows > 0) {
                        $stmtUpdate = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id = ?");
                        $stmtUpdate->bind_param("ssi", $valorStr, $clave, $cajaId);
                        $stmtUpdate->execute();
                    } else {
                        $stmtCaja = $conn->prepare("
                            INSERT INTO configuraciones (clave, valor, tipo, categoria, descripcion, caja_id, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())
                        ");
                        $stmtCaja->bind_param("sssssi", $clave, $valorStr, $tipo, $categoria, $descripcion, $cajaId);
                        $stmtCaja->execute();
                    }
                }
                // Para configuraciones de empresa (empresa_*), crear configuración global
                else if (strpos($clave, 'empresa_') === 0) {
                    // Determinar tipo basado en el valor
                    $tipo = is_numeric($valor) ? 'number' : (is_bool($valor) || $valor === 'true' || $valor === 'false' ? 'boolean' : 'string');
                    $categoria = 'empresa';
                    $descripcion = 'Configuración de empresa: ' . str_replace('_', ' ', substr($clave, 8));
                    $valorStr = cfg_convertirParaAlmacenar($valor, $tipo);

                    // Verificar si ya existe como global
                    $stmtCheck = $conn->prepare("SELECT id FROM configuraciones WHERE clave = ? AND caja_id IS NULL");
                    $stmtCheck->bind_param("s", $clave);
                    $stmtCheck->execute();
                    $checkResult = $stmtCheck->get_result();

                    if ($checkResult->num_rows > 0) {
                        // Actualizar existente
                        $stmtUpdate = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id IS NULL");
                        $stmtUpdate->bind_param("ss", $valorStr, $clave);
                        $stmtUpdate->execute();
                    } else {
                        // Crear configuración global
                        $stmtEmpresa = $conn->prepare("
                            INSERT INTO configuraciones (clave, valor, tipo, categoria, descripcion, caja_id, created_at, updated_at)
                            VALUES (?, ?, ?, ?, ?, NULL, NOW(), NOW())
                        ");
                        $stmtEmpresa->bind_param("sssss", $clave, $valorStr, $tipo, $categoria, $descripcion);
                        $stmtEmpresa->execute();
                    }
                }
                continue; // Saltar al siguiente
            }

            $configRow = $result->fetch_assoc();
            $tipo = $configRow['tipo'];
            $categoria = $configRow['categoria'];
            $descripcion = $configRow['descripcion'];

            $valorStr = cfg_convertirParaAlmacenar($valor, $tipo);

            if ($cajaId) {
                // Verificar si ya existe config para esta caja
                $stmt = $conn->prepare("SELECT id FROM configuraciones WHERE clave = ? AND caja_id = ?");
                $stmt->bind_param("si", $clave, $cajaId);
                $stmt->execute();
                $existeResult = $stmt->get_result();

                if ($existeResult->num_rows > 0) {
                    // Actualizar existente
                    $stmt = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id = ?");
                    $stmt->bind_param("ssi", $valorStr, $clave, $cajaId);
                } else {
                    // Insertar nuevo para esta caja
                    $stmt = $conn->prepare("
                        INSERT INTO configuraciones (clave, valor, tipo, categoria, descripcion, caja_id, created_at, updated_at)
                        VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())
                    ");
                    $stmt->bind_param("sssssi", $clave, $valorStr, $tipo, $categoria, $descripcion, $cajaId);
                }
            } else {
                // Actualizar config global
                $stmt = $conn->prepare("UPDATE configuraciones SET valor = ?, updated_at = NOW() WHERE clave = ? AND caja_id IS NULL");
                $stmt->bind_param("ss", $valorStr, $clave);
            }

            $stmt->execute();
        }

        $conn->commit();
        sendResponse(true, "Configuraciones actualizadas correctamente");
    } catch (Exception $e) {
        $conn->rollback();
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Registrar o actualizar una caja (por nombre_pc)
 */
function cfg_registrarCaja() {
    global $conn, $datapost;

    $identificador = isset($datapost['identificador']) ? trim($datapost['identificador']) : null;
    $nombrePc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : null;
    $ip = isset($datapost['ip']) ? $datapost['ip'] : null;

    if (!$nombrePc) {
        sendResponse(false, "Nombre de PC requerido");
        return;
    }

    try {
        // Verificar si ya existe una caja con este nombre_pc
        $stmt = $conn->prepare("SELECT id, identificador FROM cajas WHERE nombre_pc = ?");
        $stmt->bind_param("s", $nombrePc);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            // Actualizar caja existente
            $row = $result->fetch_assoc();
            $cajaId = $row['id'];

            // Solo actualizar identificador si se proporciona
            if ($identificador) {
                $stmt = $conn->prepare("UPDATE cajas SET identificador = ?, ip = ?, updated_at = NOW() WHERE id = ?");
                $stmt->bind_param("ssi", $identificador, $ip, $cajaId);
            } else {
                $stmt = $conn->prepare("UPDATE cajas SET ip = ?, updated_at = NOW() WHERE id = ?");
                $stmt->bind_param("si", $ip, $cajaId);
            }
            $stmt->execute();

            sendResponse(true, "Caja actualizada correctamente", [
                'id' => $cajaId,
                'identificador' => $identificador ?: $row['identificador'],
                'nombre_pc' => $nombrePc,
                'ip' => $ip
            ]);
        } else {
            // Insertar nueva caja
            $ident = $identificador ?: $nombrePc; // Si no hay identificador, usar nombre_pc
            $stmt = $conn->prepare("INSERT INTO cajas (identificador, nombre_pc, ip, created_at, updated_at) VALUES (?, ?, ?, NOW(), NOW())");
            $stmt->bind_param("sss", $ident, $nombrePc, $ip);
            $stmt->execute();

            sendResponse(true, "Caja registrada correctamente", [
                'id' => $conn->insert_id,
                'identificador' => $ident,
                'nombre_pc' => $nombrePc,
                'ip' => $ip
            ]);
        }
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener caja por nombre_pc
 */
function cfg_getCajaPorNombrePc() {
    global $conn, $datapost;

    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);

    if (!$nombrePc) {
        sendResponse(false, "Nombre de PC requerido");
        return;
    }

    try {
        $stmt = $conn->prepare("SELECT id, identificador, nombre_pc, ip, activo, created_at, updated_at FROM cajas WHERE nombre_pc = ?");
        $stmt->bind_param("s", $nombrePc);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($row = $result->fetch_assoc()) {
            sendResponse(true, "Caja encontrada", $row);
        } else {
            sendResponse(false, "Caja no encontrada para este equipo");
        }
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener lista de cajas registradas
 */
function cfg_getCajas() {
    global $conn;

    try {
        $query = "SELECT id, identificador, nombre_pc, ip, activo, created_at, updated_at FROM cajas ORDER BY identificador";
        $result = $conn->query($query);

        if (!$result) {
            throw new Exception("Error al obtener cajas: " . $conn->error);
        }

        $cajas = [];
        while ($row = $result->fetch_assoc()) {
            $cajas[] = $row;
        }

        sendResponse(true, "Cajas obtenidas", $cajas);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Convertir valor de string a su tipo correcto
 */
function cfg_convertirValor($valor, $tipo) {
    switch ($tipo) {
        case 'boolean':
            return $valor === 'true' || $valor === '1' || $valor === true;
        case 'number':
            return is_numeric($valor) ? (float)$valor : 0;
        case 'json':
            return json_decode($valor, true) ?? [];
        default:
            return $valor;
    }
}

/**
 * Convertir valor para almacenamiento en base de datos
 */
function cfg_convertirParaAlmacenar($valor, $tipo) {
    switch ($tipo) {
        case 'boolean':
            return ($valor === true || $valor === 'true' || $valor === 1 || $valor === '1') ? 'true' : 'false';
        case 'number':
            return (string)$valor;
        case 'json':
            return is_string($valor) ? $valor : json_encode($valor);
        default:
            return (string)$valor;
    }
}
