<?php
// Archivo: CortesCaja.php
// Descripción: API para gestionar cortes de caja de ventas de mostrador

if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'getVentasPendientesCorte':
        $caja_id = isset($datapost['caja_id']) ? intval($datapost['caja_id']) : 0;

        if ($caja_id <= 0) {
            // Intentar obtener caja_id desde nombre_pc
            $nombre_pc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : '';
            if ($nombre_pc) {
                $cajaStmt = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
                $cajaStmt->bind_param("s", $nombre_pc);
                $cajaStmt->execute();
                $cajaResult = $cajaStmt->get_result();
                if ($cajaRow = $cajaResult->fetch_assoc()) {
                    $caja_id = intval($cajaRow['id']);
                }
            }
        }

        if ($caja_id <= 0) {
            sendResponse(false, "Caja no identificada");
        }

        try {
            // Obtener ventas sin corte asignado de esta caja
            $query = "SELECT
                vm.id,
                vm.folio,
                vm.usuario_nombre,
                vm.total,
                vm.metodo_pago,
                vm.fecha_creacion,
                (SELECT COUNT(*) FROM ventas_mostrador_detalles WHERE venta_mostrador_id = vm.id) as total_productos
            FROM ventas_mostrador vm
            WHERE vm.caja_id = ? AND vm.corte_id IS NULL
            ORDER BY vm.fecha_creacion DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $caja_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $ventas = [];
            while ($row = $result->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'usuario_nombre' => $row['usuario_nombre'],
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'fecha_creacion' => $row['fecha_creacion'],
                    'total_productos' => intval($row['total_productos'])
                ];
            }

            // Calcular totales por método de pago
            $queryTotales = "SELECT
                COUNT(*) as cantidad_ventas,
                COALESCE(SUM(total), 0) as total_ventas,
                COALESCE(SUM(CASE WHEN metodo_pago = 'efectivo' THEN total ELSE 0 END), 0) as total_efectivo,
                COALESCE(SUM(CASE WHEN metodo_pago IN ('tarjeta', 'tarjeta_credito', 'tarjeta_debito') THEN total ELSE 0 END), 0) as total_tarjeta,
                COALESCE(SUM(CASE WHEN metodo_pago = 'tarjeta_credito' THEN total ELSE 0 END), 0) as total_tarjeta_credito,
                COALESCE(SUM(CASE WHEN metodo_pago = 'tarjeta_debito' THEN total ELSE 0 END), 0) as total_tarjeta_debito,
                COALESCE(SUM(CASE WHEN metodo_pago = 'transferencia' THEN total ELSE 0 END), 0) as total_transferencia
            FROM ventas_mostrador
            WHERE caja_id = ? AND corte_id IS NULL";

            $stmtTotales = $conn->prepare($queryTotales);
            $stmtTotales->bind_param("i", $caja_id);
            $stmtTotales->execute();
            $resultTotales = $stmtTotales->get_result();
            $totales = $resultTotales->fetch_assoc();

            // Para pagos mixtos, obtener desglose
            $queryMixtos = "SELECT
                vmp.metodo_pago,
                SUM(vmp.monto) as total_monto
            FROM ventas_mostrador vm
            INNER JOIN ventas_mostrador_pagos vmp ON vm.id = vmp.venta_mostrador_id
            WHERE vm.caja_id = ? AND vm.corte_id IS NULL AND vm.metodo_pago = 'mixto'
            GROUP BY vmp.metodo_pago";

            $stmtMixtos = $conn->prepare($queryMixtos);
            $stmtMixtos->bind_param("i", $caja_id);
            $stmtMixtos->execute();
            $resultMixtos = $stmtMixtos->get_result();

            while ($mixto = $resultMixtos->fetch_assoc()) {
                $metodo = $mixto['metodo_pago'];
                $monto = floatval($mixto['total_monto']);
                if ($metodo === 'efectivo') {
                    $totales['total_efectivo'] = floatval($totales['total_efectivo']) + $monto;
                } else if ($metodo === 'tarjeta' || $metodo === 'tarjeta_credito' || $metodo === 'tarjeta_debito') {
                    $totales['total_tarjeta'] = floatval($totales['total_tarjeta']) + $monto;
                    if ($metodo === 'tarjeta_credito') {
                        $totales['total_tarjeta_credito'] = floatval($totales['total_tarjeta_credito'] ?? 0) + $monto;
                    } else if ($metodo === 'tarjeta_debito') {
                        $totales['total_tarjeta_debito'] = floatval($totales['total_tarjeta_debito'] ?? 0) + $monto;
                    }
                } else if ($metodo === 'transferencia') {
                    $totales['total_transferencia'] = floatval($totales['total_transferencia']) + $monto;
                }
            }

            // Obtener fecha del último corte para esta caja
            $fechaUltimoCorte = null;
            $stmtUltimoCorte = $conn->prepare("
                SELECT fecha_creacion FROM cortes_caja
                WHERE caja_id = ?
                ORDER BY fecha_creacion DESC
                LIMIT 1
            ");
            $stmtUltimoCorte->bind_param("i", $caja_id);
            $stmtUltimoCorte->execute();
            $ultimoCorteResult = $stmtUltimoCorte->get_result();
            if ($ultimoCorteRow = $ultimoCorteResult->fetch_assoc()) {
                $fechaUltimoCorte = $ultimoCorteRow['fecha_creacion'];
            }

            // Obtener retiros de efectivo pendientes de corte
            $totalRetiros = 0;
            $cantidadRetiros = 0;
            $tablaRetirosExiste = $conn->query("SHOW TABLES LIKE 'retiros_efectivo'")->num_rows > 0;

            if ($tablaRetirosExiste) {
                if ($fechaUltimoCorte) {
                    $stmtRetiros = $conn->prepare("
                        SELECT COALESCE(SUM(monto), 0) as total_retiros, COUNT(*) as cantidad_retiros
                        FROM retiros_efectivo
                        WHERE caja_id = ? AND fecha_retiro > ?
                    ");
                    $stmtRetiros->bind_param("is", $caja_id, $fechaUltimoCorte);
                } else {
                    $stmtRetiros = $conn->prepare("
                        SELECT COALESCE(SUM(monto), 0) as total_retiros, COUNT(*) as cantidad_retiros
                        FROM retiros_efectivo
                        WHERE caja_id = ?
                    ");
                    $stmtRetiros->bind_param("i", $caja_id);
                }

                $stmtRetiros->execute();
                $retirosResult = $stmtRetiros->get_result();
                if ($retirosRow = $retirosResult->fetch_assoc()) {
                    $totalRetiros = floatval($retirosRow['total_retiros']);
                    $cantidadRetiros = intval($retirosRow['cantidad_retiros']);
                }
            }

            // Obtener ingresos de efectivo pendientes de corte
            $totalIngresos = 0;
            $cantidadIngresos = 0;
            $tablaIngresosExiste = $conn->query("SHOW TABLES LIKE 'ingresos_efectivo'")->num_rows > 0;

            if ($tablaIngresosExiste) {
                if ($fechaUltimoCorte) {
                    $stmtIngresos = $conn->prepare("
                        SELECT COALESCE(SUM(monto), 0) as total_ingresos, COUNT(*) as cantidad_ingresos
                        FROM ingresos_efectivo
                        WHERE caja_id = ? AND fecha_ingreso > ?
                    ");
                    $stmtIngresos->bind_param("is", $caja_id, $fechaUltimoCorte);
                } else {
                    $stmtIngresos = $conn->prepare("
                        SELECT COALESCE(SUM(monto), 0) as total_ingresos, COUNT(*) as cantidad_ingresos
                        FROM ingresos_efectivo
                        WHERE caja_id = ?
                    ");
                    $stmtIngresos->bind_param("i", $caja_id);
                }

                $stmtIngresos->execute();
                $ingresosResult = $stmtIngresos->get_result();
                if ($ingresosRow = $ingresosResult->fetch_assoc()) {
                    $totalIngresos = floatval($ingresosRow['total_ingresos']);
                    $cantidadIngresos = intval($ingresosRow['cantidad_ingresos']);
                }
            }

            // Calcular efectivo neto: ventas + ingresos - retiros
            $efectivoNeto = floatval($totales['total_efectivo']) + $totalIngresos - $totalRetiros;

            sendResponse(true, "Ventas pendientes obtenidas", [
                'ventas' => $ventas,
                'totales' => [
                    'cantidad_ventas' => intval($totales['cantidad_ventas']),
                    'total_ventas' => floatval($totales['total_ventas']),
                    'total_efectivo' => $efectivoNeto,
                    'total_efectivo_bruto' => floatval($totales['total_efectivo']),
                    'total_retiros' => $totalRetiros,
                    'cantidad_retiros' => $cantidadRetiros,
                    'total_ingresos' => $totalIngresos,
                    'cantidad_ingresos' => $cantidadIngresos,
                    'total_tarjeta' => floatval($totales['total_tarjeta']),
                    'total_tarjeta_credito' => floatval($totales['total_tarjeta_credito'] ?? 0),
                    'total_tarjeta_debito' => floatval($totales['total_tarjeta_debito'] ?? 0),
                    'total_transferencia' => floatval($totales['total_transferencia'])
                ],
                'caja_id' => $caja_id
            ]);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener ventas pendientes: " . $e->getMessage());
        }
        break;

    case 'crearCorte':
        $caja_id = isset($datapost['caja_id']) ? intval($datapost['caja_id']) : 0;
        $usuario_id = intval($datapost['usuario_id'] ?? 0);
        $usuario_nombre = trim($datapost['usuario_nombre'] ?? '');
        $fondo_inicial = floatval($datapost['fondo_inicial'] ?? 0);
        $efectivo_contado = floatval($datapost['efectivo_contado'] ?? 0);
        $observaciones = trim($datapost['observaciones'] ?? '');

        // Intentar obtener caja_id desde nombre_pc si no viene
        if ($caja_id <= 0) {
            $nombre_pc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : '';
            if ($nombre_pc) {
                $cajaStmt = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
                $cajaStmt->bind_param("s", $nombre_pc);
                $cajaStmt->execute();
                $cajaResult = $cajaStmt->get_result();
                if ($cajaRow = $cajaResult->fetch_assoc()) {
                    $caja_id = intval($cajaRow['id']);
                }
            }
        }

        if ($caja_id <= 0) {
            sendResponse(false, "Caja no identificada");
        }

        if ($usuario_id <= 0) {
            sendResponse(false, "Usuario requerido");
        }

        try {
            $conn->begin_transaction();

            // Calcular totales de ventas pendientes
            $queryTotales = "SELECT
                COUNT(*) as cantidad_ventas,
                COALESCE(SUM(total), 0) as total_ventas,
                COALESCE(SUM(CASE WHEN metodo_pago = 'efectivo' THEN total ELSE 0 END), 0) as total_efectivo,
                COALESCE(SUM(CASE WHEN metodo_pago IN ('tarjeta', 'tarjeta_credito', 'tarjeta_debito') THEN total ELSE 0 END), 0) as total_tarjeta,
                COALESCE(SUM(CASE WHEN metodo_pago = 'tarjeta_credito' THEN total ELSE 0 END), 0) as total_tarjeta_credito,
                COALESCE(SUM(CASE WHEN metodo_pago = 'tarjeta_debito' THEN total ELSE 0 END), 0) as total_tarjeta_debito,
                COALESCE(SUM(CASE WHEN metodo_pago = 'transferencia' THEN total ELSE 0 END), 0) as total_transferencia
            FROM ventas_mostrador
            WHERE caja_id = ? AND corte_id IS NULL";

            $stmtTotales = $conn->prepare($queryTotales);
            $stmtTotales->bind_param("i", $caja_id);
            $stmtTotales->execute();
            $resultTotales = $stmtTotales->get_result();
            $totales = $resultTotales->fetch_assoc();

            // Para pagos mixtos, obtener desglose
            $queryMixtos = "SELECT
                vmp.metodo_pago,
                SUM(vmp.monto) as total_monto
            FROM ventas_mostrador vm
            INNER JOIN ventas_mostrador_pagos vmp ON vm.id = vmp.venta_mostrador_id
            WHERE vm.caja_id = ? AND vm.corte_id IS NULL AND vm.metodo_pago = 'mixto'
            GROUP BY vmp.metodo_pago";

            $stmtMixtos = $conn->prepare($queryMixtos);
            $stmtMixtos->bind_param("i", $caja_id);
            $stmtMixtos->execute();
            $resultMixtos = $stmtMixtos->get_result();

            $total_efectivo = floatval($totales['total_efectivo']);
            $total_tarjeta = floatval($totales['total_tarjeta']);
            $total_tarjeta_credito = floatval($totales['total_tarjeta_credito']);
            $total_tarjeta_debito = floatval($totales['total_tarjeta_debito']);
            $total_transferencia = floatval($totales['total_transferencia']);

            while ($mixto = $resultMixtos->fetch_assoc()) {
                $metodo = $mixto['metodo_pago'];
                $monto = floatval($mixto['total_monto']);
                if ($metodo === 'efectivo') {
                    $total_efectivo += $monto;
                } else if ($metodo === 'tarjeta' || $metodo === 'tarjeta_credito' || $metodo === 'tarjeta_debito') {
                    $total_tarjeta += $monto;
                    if ($metodo === 'tarjeta_credito') {
                        $total_tarjeta_credito += $monto;
                    } else if ($metodo === 'tarjeta_debito') {
                        $total_tarjeta_debito += $monto;
                    }
                } else if ($metodo === 'transferencia') {
                    $total_transferencia += $monto;
                }
            }

            $cantidad_ventas = intval($totales['cantidad_ventas']);
            $total_ventas = floatval($totales['total_ventas']);

            // Obtener fecha del último corte
            $fechaUltimoCorte = null;
            $stmtUltimoCorte = $conn->prepare("
                SELECT fecha_creacion FROM cortes_caja
                WHERE caja_id = ?
                ORDER BY fecha_creacion DESC
                LIMIT 1
            ");
            $stmtUltimoCorte->bind_param("i", $caja_id);
            $stmtUltimoCorte->execute();
            $ultimoCorteResult = $stmtUltimoCorte->get_result();
            if ($ultimoCorteRow = $ultimoCorteResult->fetch_assoc()) {
                $fechaUltimoCorte = $ultimoCorteRow['fecha_creacion'];
            }

            // Obtener ingresos de efectivo pendientes
            $total_ingresos = 0;
            $tablaIngresosExiste = $conn->query("SHOW TABLES LIKE 'ingresos_efectivo'")->num_rows > 0;
            if ($tablaIngresosExiste) {
                if ($fechaUltimoCorte) {
                    $stmtIngresos = $conn->prepare("SELECT COALESCE(SUM(monto), 0) as total FROM ingresos_efectivo WHERE caja_id = ? AND fecha_ingreso > ?");
                    $stmtIngresos->bind_param("is", $caja_id, $fechaUltimoCorte);
                } else {
                    $stmtIngresos = $conn->prepare("SELECT COALESCE(SUM(monto), 0) as total FROM ingresos_efectivo WHERE caja_id = ?");
                    $stmtIngresos->bind_param("i", $caja_id);
                }
                $stmtIngresos->execute();
                $ingresosResult = $stmtIngresos->get_result();
                if ($ingresosRow = $ingresosResult->fetch_assoc()) {
                    $total_ingresos = floatval($ingresosRow['total']);
                }
            }

            // Obtener retiros de efectivo pendientes
            $total_retiros = 0;
            $tablaRetirosExiste = $conn->query("SHOW TABLES LIKE 'retiros_efectivo'")->num_rows > 0;
            if ($tablaRetirosExiste) {
                if ($fechaUltimoCorte) {
                    $stmtRetiros = $conn->prepare("SELECT COALESCE(SUM(monto), 0) as total FROM retiros_efectivo WHERE caja_id = ? AND fecha_retiro > ?");
                    $stmtRetiros->bind_param("is", $caja_id, $fechaUltimoCorte);
                } else {
                    $stmtRetiros = $conn->prepare("SELECT COALESCE(SUM(monto), 0) as total FROM retiros_efectivo WHERE caja_id = ?");
                    $stmtRetiros->bind_param("i", $caja_id);
                }
                $stmtRetiros->execute();
                $retirosResult = $stmtRetiros->get_result();
                if ($retirosRow = $retirosResult->fetch_assoc()) {
                    $total_retiros = floatval($retirosRow['total']);
                }
            }

            // Calcular esperado y diferencia (fondo + ventas efectivo + ingresos - retiros)
            $efectivo_esperado = $fondo_inicial + $total_efectivo + $total_ingresos - $total_retiros;
            $diferencia = $efectivo_contado - $efectivo_esperado;

            // Insertar corte
            $insertCorte = "INSERT INTO cortes_caja (
                caja_id, usuario_id, usuario_nombre,
                fondo_inicial, total_efectivo, total_tarjeta, total_tarjeta_credito, total_tarjeta_debito, total_transferencia,
                total_ventas, cantidad_ventas,
                efectivo_contado, efectivo_esperado, diferencia,
                observaciones, total_ingresos, total_retiros
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

            $stmtInsert = $conn->prepare($insertCorte);
            $stmtInsert->bind_param(
                "iisddddddiiddsdd",
                $caja_id,
                $usuario_id,
                $usuario_nombre,
                $fondo_inicial,
                $total_efectivo,
                $total_tarjeta,
                $total_tarjeta_credito,
                $total_tarjeta_debito,
                $total_transferencia,
                $total_ventas,
                $cantidad_ventas,
                $efectivo_contado,
                $efectivo_esperado,
                $diferencia,
                $observaciones,
                $total_ingresos,
                $total_retiros
            );

            if (!$stmtInsert->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al crear corte: " . $stmtInsert->error);
            }

            $corte_id = $conn->insert_id;

            // Asociar ventas al corte
            $updateVentas = "UPDATE ventas_mostrador SET corte_id = ? WHERE caja_id = ? AND corte_id IS NULL";
            $stmtUpdate = $conn->prepare($updateVentas);
            $stmtUpdate->bind_param("ii", $corte_id, $caja_id);
            $stmtUpdate->execute();

            $conn->commit();

            sendResponse(true, "Corte creado exitosamente", [
                'corte_id' => $corte_id,
                'fondo_inicial' => $fondo_inicial,
                'total_efectivo' => $total_efectivo,
                'total_tarjeta' => $total_tarjeta,
                'total_tarjeta_credito' => $total_tarjeta_credito,
                'total_tarjeta_debito' => $total_tarjeta_debito,
                'total_transferencia' => $total_transferencia,
                'total_ventas' => $total_ventas,
                'cantidad_ventas' => $cantidad_ventas,
                'efectivo_contado' => $efectivo_contado,
                'efectivo_esperado' => $efectivo_esperado,
                'diferencia' => $diferencia,
                'total_ingresos' => $total_ingresos,
                'total_retiros' => $total_retiros
            ]);
        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al crear corte: " . $e->getMessage());
        }
        break;

    case 'getHistorialCortes':
        $caja_id = isset($datapost['caja_id']) ? intval($datapost['caja_id']) : 0;
        $limite = isset($datapost['limite']) ? intval($datapost['limite']) : 20;
        $offset = isset($datapost['offset']) ? intval($datapost['offset']) : 0;

        // Intentar obtener caja_id desde nombre_pc si no viene
        if ($caja_id <= 0) {
            $nombre_pc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : '';
            if ($nombre_pc) {
                $cajaStmt = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
                $cajaStmt->bind_param("s", $nombre_pc);
                $cajaStmt->execute();
                $cajaResult = $cajaStmt->get_result();
                if ($cajaRow = $cajaResult->fetch_assoc()) {
                    $caja_id = intval($cajaRow['id']);
                }
            }
        }

        if ($caja_id <= 0) {
            sendResponse(false, "Caja no identificada");
        }

        try {
            $query = "SELECT * FROM cortes_caja WHERE caja_id = ? ORDER BY fecha_creacion DESC LIMIT ? OFFSET ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("iii", $caja_id, $limite, $offset);
            $stmt->execute();
            $result = $stmt->get_result();

            $cortes = [];
            while ($row = $result->fetch_assoc()) {
                $cortes[] = [
                    'id' => intval($row['id']),
                    'usuario_nombre' => $row['usuario_nombre'],
                    'fondo_inicial' => floatval($row['fondo_inicial']),
                    'total_efectivo' => floatval($row['total_efectivo']),
                    'total_tarjeta' => floatval($row['total_tarjeta']),
                    'total_tarjeta_credito' => floatval($row['total_tarjeta_credito'] ?? 0),
                    'total_tarjeta_debito' => floatval($row['total_tarjeta_debito'] ?? 0),
                    'total_transferencia' => floatval($row['total_transferencia']),
                    'total_ventas' => floatval($row['total_ventas']),
                    'cantidad_ventas' => intval($row['cantidad_ventas']),
                    'efectivo_contado' => floatval($row['efectivo_contado']),
                    'efectivo_esperado' => floatval($row['efectivo_esperado']),
                    'diferencia' => floatval($row['diferencia']),
                    'total_ingresos' => floatval($row['total_ingresos'] ?? 0),
                    'total_retiros' => floatval($row['total_retiros'] ?? 0),
                    'observaciones' => $row['observaciones'],
                    'fecha_creacion' => $row['fecha_creacion']
                ];
            }

            // Contar total de cortes
            $queryCount = "SELECT COUNT(*) as total FROM cortes_caja WHERE caja_id = ?";
            $stmtCount = $conn->prepare($queryCount);
            $stmtCount->bind_param("i", $caja_id);
            $stmtCount->execute();
            $resultCount = $stmtCount->get_result();
            $total = $resultCount->fetch_assoc()['total'];

            sendResponse(true, "Historial de cortes obtenido", [
                'cortes' => $cortes,
                'total' => intval($total)
            ]);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener historial: " . $e->getMessage());
        }
        break;

    case 'getDetalleCorte':
        $corte_id = intval($datapost['corte_id'] ?? 0);

        if ($corte_id <= 0) {
            sendResponse(false, "ID de corte requerido");
        }

        try {
            // Obtener datos del corte
            $query = "SELECT c.*, ca.identificador
                      FROM cortes_caja c
                      LEFT JOIN cajas ca ON c.caja_id = ca.id
                      WHERE c.id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $corte_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows === 0) {
                sendResponse(false, "Corte no encontrado");
            }

            $corte = $result->fetch_assoc();

            // Obtener ventas del corte
            $queryVentas = "SELECT
                vm.id, vm.folio, vm.usuario_nombre, vm.total, vm.metodo_pago, vm.fecha_creacion
            FROM ventas_mostrador vm
            WHERE vm.corte_id = ?
            ORDER BY vm.fecha_creacion ASC";

            $stmtVentas = $conn->prepare($queryVentas);
            $stmtVentas->bind_param("i", $corte_id);
            $stmtVentas->execute();
            $resultVentas = $stmtVentas->get_result();

            $ventas = [];
            while ($row = $resultVentas->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'usuario_nombre' => $row['usuario_nombre'],
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'fecha_creacion' => $row['fecha_creacion']
                ];
            }

            sendResponse(true, "Detalle de corte obtenido", [
                'corte' => [
                    'id' => intval($corte['id']),
                    'caja_id' => intval($corte['caja_id']),
                    'identificador' => $corte['identificador'],
                    'usuario_nombre' => $corte['usuario_nombre'],
                    'fondo_inicial' => floatval($corte['fondo_inicial']),
                    'total_efectivo' => floatval($corte['total_efectivo']),
                    'total_tarjeta' => floatval($corte['total_tarjeta']),
                    'total_tarjeta_credito' => floatval($corte['total_tarjeta_credito'] ?? 0),
                    'total_tarjeta_debito' => floatval($corte['total_tarjeta_debito'] ?? 0),
                    'total_transferencia' => floatval($corte['total_transferencia']),
                    'total_ventas' => floatval($corte['total_ventas']),
                    'cantidad_ventas' => intval($corte['cantidad_ventas']),
                    'efectivo_contado' => floatval($corte['efectivo_contado']),
                    'efectivo_esperado' => floatval($corte['efectivo_esperado']),
                    'diferencia' => floatval($corte['diferencia']),
                    'total_ingresos' => floatval($corte['total_ingresos'] ?? 0),
                    'total_retiros' => floatval($corte['total_retiros'] ?? 0),
                    'observaciones' => $corte['observaciones'],
                    'fecha_creacion' => $corte['fecha_creacion']
                ],
                'ventas' => $ventas
            ]);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener detalle: " . $e->getMessage());
        }
        break;

    case 'imprimirCorte':
        $corte_id = intval($datapost['corte_id'] ?? 0);

        if ($corte_id <= 0) {
            sendResponse(false, "ID de corte requerido");
        }

        try {
            // Obtener datos del corte
            $query = "SELECT c.*, ca.identificador
                      FROM cortes_caja c
                      LEFT JOIN cajas ca ON c.caja_id = ca.id
                      WHERE c.id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $corte_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows === 0) {
                sendResponse(false, "Corte no encontrado");
            }

            $corte = $result->fetch_assoc();

            // Generar HTML para impresión
            $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Corte de Caja #' . $corte['id'] . '</title>
    <style>
        body {
            font-family: "Courier New", monospace;
            font-size: 12px;
            width: 80mm;
            margin: 0 auto;
            padding: 5mm;
        }
        .header {
            text-align: center;
            margin-bottom: 10px;
            border-bottom: 1px dashed #000;
            padding-bottom: 10px;
        }
        .title {
            font-size: 16px;
            font-weight: bold;
        }
        .row {
            display: flex;
            justify-content: space-between;
            margin: 4px 0;
        }
        .label {
            font-weight: normal;
        }
        .value {
            font-weight: bold;
        }
        .section {
            margin: 10px 0;
            padding: 5px 0;
            border-top: 1px dashed #000;
        }
        .total-row {
            font-size: 14px;
            font-weight: bold;
            margin-top: 10px;
        }
        .diferencia-positiva {
            color: green;
        }
        .diferencia-negativa {
            color: red;
        }
        @media print {
            @page { margin: 0; size: 80mm auto; }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="title">CORTE DE CAJA</div>
        <div>Corte #' . $corte['id'] . '</div>
        <div>' . date('d/m/Y H:i', strtotime($corte['fecha_creacion'])) . '</div>
    </div>

    <div class="row">
        <span class="label">Identificador:</span>
        <span class="value">' . htmlspecialchars($corte['identificador'] ?? 'N/A') . '</span>
    </div>
    <div class="row">
        <span class="label">Cajero:</span>
        <span class="value">' . htmlspecialchars($corte['usuario_nombre']) . '</span>
    </div>

    <div class="section">
        <div class="row">
            <span class="label">Ventas realizadas:</span>
            <span class="value">' . $corte['cantidad_ventas'] . '</span>
        </div>
    </div>

    <div class="section">
        <div class="row">
            <span class="label">Efectivo:</span>
            <span class="value">$' . number_format($corte['total_efectivo'], 2) . '</span>
        </div>
        <div class="row">
            <span class="label">T. Crédito:</span>
            <span class="value">$' . number_format($corte['total_tarjeta_credito'] ?? 0, 2) . '</span>
        </div>
        <div class="row">
            <span class="label">T. Débito:</span>
            <span class="value">$' . number_format($corte['total_tarjeta_debito'] ?? 0, 2) . '</span>
        </div>
        <div class="row">
            <span class="label">Transferencia:</span>
            <span class="value">$' . number_format($corte['total_transferencia'], 2) . '</span>
        </div>
        <div class="row total-row">
            <span>TOTAL VENTAS:</span>
            <span>$' . number_format($corte['total_ventas'], 2) . '</span>
        </div>
    </div>

    <div class="section">
        <div class="row">
            <span class="label">Fondo Inicial:</span>
            <span class="value">$' . number_format($corte['fondo_inicial'], 2) . '</span>
        </div>
        <div class="row">
            <span class="label">+ Efectivo Ventas:</span>
            <span class="value">$' . number_format($corte['total_efectivo'], 2) . '</span>
        </div>
        ' . (isset($corte['total_ingresos']) && $corte['total_ingresos'] > 0 ? '
        <div class="row">
            <span class="label">+ Ingresos:</span>
            <span class="value" style="color: green;">$' . number_format($corte['total_ingresos'], 2) . '</span>
        </div>' : '') . '
        ' . (isset($corte['total_retiros']) && $corte['total_retiros'] > 0 ? '
        <div class="row">
            <span class="label">- Retiros:</span>
            <span class="value" style="color: red;">$' . number_format($corte['total_retiros'], 2) . '</span>
        </div>' : '') . '
        <div class="row total-row">
            <span>ESPERADO EN CAJA:</span>
            <span>$' . number_format($corte['efectivo_esperado'], 2) . '</span>
        </div>
    </div>

    <div class="section">
        <div class="row total-row">
            <span>EFECTIVO CONTADO:</span>
            <span>$' . number_format($corte['efectivo_contado'], 2) . '</span>
        </div>
        <div class="row total-row">
            <span>DIFERENCIA:</span>
            <span class="' . ($corte['diferencia'] >= 0 ? 'diferencia-positiva' : 'diferencia-negativa') . '">
                $' . number_format($corte['diferencia'], 2) . '
            </span>
        </div>
    </div>

    ' . ($corte['observaciones'] ? '
    <div class="section">
        <div><strong>Observaciones:</strong></div>
        <div>' . htmlspecialchars($corte['observaciones']) . '</div>
    </div>
    ' : '') . '

    <div class="section" style="text-align: center; margin-top: 20px;">
        <div>________________________________</div>
        <div>Firma del Cajero</div>
    </div>
</body>
</html>';

            sendResponse(true, "Ticket generado", [
                'html' => $html,
                'corte_id' => $corte_id
            ]);
        } catch (Exception $e) {
            sendResponse(false, "Error al generar ticket: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para cortes de caja");
        break;
}
?>
