<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'getDescuentosVolumen':
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            $query = "SELECT
                id,
                producto_id,
                cantidad_minima,
                descuento_porcentaje,
                activo
            FROM descuentos_volumen
            WHERE producto_id = ? AND activo = 1
            ORDER BY cantidad_minima ASC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $producto_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $descuentos = [];
            while ($row = $result->fetch_assoc()) {
                $descuentos[] = [
                    'id' => intval($row['id']),
                    'producto_id' => intval($row['producto_id']),
                    'cantidad_minima' => floatval($row['cantidad_minima']),
                    'descuento_porcentaje' => floatval($row['descuento_porcentaje']),
                    'activo' => intval($row['activo'])
                ];
            }

            sendResponse(true, "Descuentos obtenidos correctamente", $descuentos);
        } catch (Exception $e) {
            error_log("Error al obtener descuentos por volumen: " . $e->getMessage());
            sendResponse(false, "Error al obtener descuentos por volumen");
        }
        break;

    case 'guardarDescuentosVolumen':
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;
            $descuentos = isset($datapost['descuentos']) ? $datapost['descuentos'] : [];

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            // Verificar que el producto existe
            $checkQuery = "SELECT id FROM productos WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $producto_id);
            $checkStmt->execute();

            if ($checkStmt->get_result()->num_rows === 0) {
                sendResponse(false, "Producto no encontrado");
                break;
            }

            // Iniciar transacción
            $conn->begin_transaction();

            try {
                // Desactivar todos los descuentos existentes del producto
                $deleteQuery = "UPDATE descuentos_volumen SET activo = 0 WHERE producto_id = ?";
                $deleteStmt = $conn->prepare($deleteQuery);
                $deleteStmt->bind_param("i", $producto_id);
                $deleteStmt->execute();

                // Insertar o actualizar descuentos
                $insertQuery = "INSERT INTO descuentos_volumen
                    (producto_id, cantidad_minima, descuento_porcentaje, activo)
                    VALUES (?, ?, ?, 1)
                    ON DUPLICATE KEY UPDATE
                    descuento_porcentaje = VALUES(descuento_porcentaje),
                    activo = 1,
                    fecha_actualizacion = CURRENT_TIMESTAMP";

                $insertStmt = $conn->prepare($insertQuery);

                foreach ($descuentos as $descuento) {
                    $cantidad_minima = floatval($descuento['cantidad_minima']);
                    $descuento_porcentaje = floatval($descuento['descuento_porcentaje']);

                    // Validaciones
                    if ($cantidad_minima <= 0) {
                        throw new Exception("La cantidad mínima debe ser mayor a 0");
                    }

                    if ($descuento_porcentaje < 0 || $descuento_porcentaje > 100) {
                        throw new Exception("El porcentaje de descuento debe estar entre 0 y 100");
                    }

                    $insertStmt->bind_param("idd", $producto_id, $cantidad_minima, $descuento_porcentaje);
                    $insertStmt->execute();
                }

                $conn->commit();
                sendResponse(true, "Descuentos guardados correctamente");
            } catch (Exception $e) {
                $conn->rollback();
                throw $e;
            }
        } catch (Exception $e) {
            error_log("Error al guardar descuentos por volumen: " . $e->getMessage());
            sendResponse(false, "Error al guardar descuentos: " . $e->getMessage());
        }
        break;

    case 'eliminarDescuentoVolumen':
        try {
            $id = isset($datapost['id']) ? intval($datapost['id']) : 0;

            if ($id <= 0) {
                sendResponse(false, "ID de descuento requerido");
                break;
            }

            $query = "UPDATE descuentos_volumen SET activo = 0 WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $id);

            if ($stmt->execute()) {
                sendResponse(true, "Descuento eliminado correctamente");
            } else {
                sendResponse(false, "Error al eliminar descuento");
            }
        } catch (Exception $e) {
            error_log("Error al eliminar descuento por volumen: " . $e->getMessage());
            sendResponse(false, "Error al eliminar descuento");
        }
        break;

    case 'calcularPrecioConDescuento':
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;
            $cantidad = isset($datapost['cantidad']) ? floatval($datapost['cantidad']) : 0;

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            if ($cantidad <= 0) {
                sendResponse(false, "Cantidad requerida");
                break;
            }

            // Obtener precio_lista del producto
            $productoQuery = "SELECT id, nombre, precio_lista FROM productos WHERE id = ? AND activo = 1";
            $productoStmt = $conn->prepare($productoQuery);
            $productoStmt->bind_param("i", $producto_id);
            $productoStmt->execute();
            $productoResult = $productoStmt->get_result();

            if ($productoResult->num_rows === 0) {
                sendResponse(false, "Producto no encontrado");
                break;
            }

            $producto = $productoResult->fetch_assoc();
            $precio_lista = floatval($producto['precio_lista']);

            // Buscar el descuento aplicable (el mayor rango que cumpla la condición)
            $descuentoQuery = "SELECT
                id,
                cantidad_minima,
                descuento_porcentaje
            FROM descuentos_volumen
            WHERE producto_id = ?
                AND activo = 1
                AND cantidad_minima <= ?
            ORDER BY cantidad_minima DESC
            LIMIT 1";

            $descuentoStmt = $conn->prepare($descuentoQuery);
            $descuentoStmt->bind_param("id", $producto_id, $cantidad);
            $descuentoStmt->execute();
            $descuentoResult = $descuentoStmt->get_result();

            $descuento_aplicado = 0;
            $cantidad_minima_aplicada = 0;
            $descuento_id = null;

            if ($descuentoResult->num_rows > 0) {
                $descuento = $descuentoResult->fetch_assoc();
                $descuento_aplicado = floatval($descuento['descuento_porcentaje']);
                $cantidad_minima_aplicada = floatval($descuento['cantidad_minima']);
                $descuento_id = intval($descuento['id']);
            }

            // Calcular precio final
            $precio_final = $precio_lista * (1 - ($descuento_aplicado / 100));

            sendResponse(true, "Precio calculado correctamente", [
                'producto_id' => intval($producto['id']),
                'producto_nombre' => $producto['nombre'],
                'precio_lista' => $precio_lista,
                'cantidad' => $cantidad,
                'descuento_id' => $descuento_id,
                'cantidad_minima_aplicada' => $cantidad_minima_aplicada,
                'descuento_porcentaje' => $descuento_aplicado,
                'precio_final' => round($precio_final, 2),
                'ahorro_unitario' => round($precio_lista - $precio_final, 2),
                'ahorro_total' => round(($precio_lista - $precio_final) * $cantidad, 2)
            ]);
        } catch (Exception $e) {
            error_log("Error al calcular precio con descuento: " . $e->getMessage());
            sendResponse(false, "Error al calcular precio");
        }
        break;

    case 'updateDescuentoVolumen':
        // Actualizar o crear la cantidad mínima y porcentaje del descuento principal de un producto
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;
            $cantidad_minima = isset($datapost['cantidad_minima']) ? floatval($datapost['cantidad_minima']) : 0;
            $descuento_porcentaje = isset($datapost['descuento_porcentaje']) ? floatval($datapost['descuento_porcentaje']) : null;

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            // Si cantidad es 0, desactivar descuentos del producto
            if ($cantidad_minima <= 0) {
                $query = "UPDATE descuentos_volumen SET activo = 0 WHERE producto_id = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("i", $producto_id);
                $stmt->execute();
                sendResponse(true, "Descuento desactivado");
                break;
            }

            // Buscar si ya existe un descuento para este producto
            $checkQuery = "SELECT id, descuento_porcentaje FROM descuentos_volumen WHERE producto_id = ? AND activo = 1 ORDER BY cantidad_minima ASC LIMIT 1";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $producto_id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows > 0) {
                // Actualizar el descuento existente
                $existente = $checkResult->fetch_assoc();

                // Si se envió porcentaje, actualizar ambos. Si no, solo cantidad.
                if ($descuento_porcentaje !== null) {
                    $updateQuery = "UPDATE descuentos_volumen SET cantidad_minima = ?, descuento_porcentaje = ?, fecha_actualizacion = CURRENT_TIMESTAMP WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->bind_param("ddi", $cantidad_minima, $descuento_porcentaje, $existente['id']);
                } else {
                    $updateQuery = "UPDATE descuentos_volumen SET cantidad_minima = ?, fecha_actualizacion = CURRENT_TIMESTAMP WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->bind_param("di", $cantidad_minima, $existente['id']);
                }
                $updateStmt->execute();
            } else {
                // Crear nuevo descuento
                $porcentaje = $descuento_porcentaje !== null ? $descuento_porcentaje : 0;
                $insertQuery = "INSERT INTO descuentos_volumen (producto_id, cantidad_minima, descuento_porcentaje, activo) VALUES (?, ?, ?, 1)";
                $insertStmt = $conn->prepare($insertQuery);
                $insertStmt->bind_param("idd", $producto_id, $cantidad_minima, $porcentaje);
                $insertStmt->execute();
            }

            sendResponse(true, "Descuento actualizado correctamente");
        } catch (Exception $e) {
            error_log("Error al actualizar descuento volumen: " . $e->getMessage());
            sendResponse(false, "Error al actualizar descuento");
        }
        break;

    case 'getDescuentosVolumenMultiples':
        // Obtener descuentos de múltiples productos a la vez (útil para el carrito)
        try {
            $producto_ids = isset($datapost['producto_ids']) ? $datapost['producto_ids'] : [];

            if (empty($producto_ids) || !is_array($producto_ids)) {
                sendResponse(false, "IDs de productos requeridos");
                break;
            }

            // Sanitizar IDs
            $producto_ids = array_map('intval', $producto_ids);
            $placeholders = implode(',', array_fill(0, count($producto_ids), '?'));

            $query = "SELECT
                dv.id,
                dv.producto_id,
                dv.cantidad_minima,
                dv.descuento_porcentaje
            FROM descuentos_volumen dv
            WHERE dv.producto_id IN ($placeholders) AND dv.activo = 1
            ORDER BY dv.producto_id, dv.cantidad_minima ASC";

            $stmt = $conn->prepare($query);

            // Bind dinámico para múltiples parámetros
            $types = str_repeat('i', count($producto_ids));
            $stmt->bind_param($types, ...$producto_ids);
            $stmt->execute();
            $result = $stmt->get_result();

            // Agrupar por producto
            $descuentosPorProducto = [];
            while ($row = $result->fetch_assoc()) {
                $pid = intval($row['producto_id']);
                if (!isset($descuentosPorProducto[$pid])) {
                    $descuentosPorProducto[$pid] = [];
                }
                $descuentosPorProducto[$pid][] = [
                    'id' => intval($row['id']),
                    'cantidad_minima' => floatval($row['cantidad_minima']),
                    'descuento_porcentaje' => floatval($row['descuento_porcentaje'])
                ];
            }

            sendResponse(true, "Descuentos obtenidos correctamente", $descuentosPorProducto);
        } catch (Exception $e) {
            error_log("Error al obtener descuentos múltiples: " . $e->getMessage());
            sendResponse(false, "Error al obtener descuentos");
        }
        break;

    default:
        sendResponse(false, "Operación no válida para descuentos por volumen");
        break;
}
?>
