<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    // =====================================================
    // CATÁLOGOS SAT
    // =====================================================

    case 'getCatalogoRegimenFiscal':
        try {
            $query = "SELECT clave, descripcion, persona_fisica, persona_moral
                      FROM cat_regimen_fiscal
                      WHERE activo = 1
                      ORDER BY clave ASC";
            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error al obtener catálogo: " . $conn->error);
            }

            $regimenes = [];
            while ($row = $result->fetch_assoc()) {
                $regimenes[] = [
                    'clave' => $row['clave'],
                    'descripcion' => $row['descripcion'],
                    'persona_fisica' => (bool)$row['persona_fisica'],
                    'persona_moral' => (bool)$row['persona_moral']
                ];
            }

            sendResponse(true, "Catálogo obtenido", $regimenes);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    case 'getCatalogoUsoCFDI':
        try {
            $personaTipo = $datapost['persona_tipo'] ?? null;

            $query = "SELECT clave, descripcion, persona_fisica, persona_moral
                      FROM cat_uso_cfdi
                      WHERE activo = 1";

            if ($personaTipo === 'fisica') {
                $query .= " AND persona_fisica = 1";
            } elseif ($personaTipo === 'moral') {
                $query .= " AND persona_moral = 1";
            }

            $query .= " ORDER BY clave ASC";
            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error al obtener catálogo: " . $conn->error);
            }

            $usos = [];
            while ($row = $result->fetch_assoc()) {
                $usos[] = [
                    'clave' => $row['clave'],
                    'descripcion' => $row['descripcion'],
                    'persona_fisica' => (bool)$row['persona_fisica'],
                    'persona_moral' => (bool)$row['persona_moral']
                ];
            }

            sendResponse(true, "Catálogo obtenido", $usos);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    case 'getCatalogoFormaPago':
        try {
            $query = "SELECT clave, descripcion
                      FROM cat_forma_pago
                      WHERE activo = 1
                      ORDER BY clave ASC";
            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error al obtener catálogo: " . $conn->error);
            }

            $formas = [];
            while ($row = $result->fetch_assoc()) {
                $formas[] = [
                    'clave' => $row['clave'],
                    'descripcion' => $row['descripcion']
                ];
            }

            sendResponse(true, "Catálogo obtenido", $formas);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    case 'getCatalogoMetodoPago':
        try {
            $query = "SELECT clave, descripcion
                      FROM cat_metodo_pago
                      WHERE activo = 1
                      ORDER BY clave ASC";
            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error al obtener catálogo: " . $conn->error);
            }

            $metodos = [];
            while ($row = $result->fetch_assoc()) {
                $metodos[] = [
                    'clave' => $row['clave'],
                    'descripcion' => $row['descripcion']
                ];
            }

            sendResponse(true, "Catálogo obtenido", $metodos);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // DATOS FISCALES DEL CLIENTE
    // =====================================================

    case 'getDatosFiscalesCliente':
        $cliente_id = intval($datapost['cliente_id'] ?? 0);

        if ($cliente_id <= 0) {
            sendResponse(false, "ID de cliente requerido");
        }

        try {
            $query = "SELECT id, nombre, rfc, razon_social, regimen_fiscal, uso_cfdi,
                             codigo_postal_fiscal, email_facturacion, requiere_factura
                      FROM clientes
                      WHERE id = ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $cliente_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                $datosFiscales = [
                    'cliente_id' => intval($row['id']),
                    'nombre' => $row['nombre'],
                    'rfc' => $row['rfc'] ?? '',
                    'razon_social' => $row['razon_social'] ?? '',
                    'regimen_fiscal' => $row['regimen_fiscal'] ?? '',
                    'uso_cfdi' => $row['uso_cfdi'] ?? '',
                    'codigo_postal_fiscal' => $row['codigo_postal_fiscal'] ?? '',
                    'email_facturacion' => $row['email_facturacion'] ?? '',
                    'requiere_factura' => (bool)($row['requiere_factura'] ?? 0),
                    'datos_completos' => !empty($row['rfc']) && !empty($row['razon_social']) &&
                                         !empty($row['regimen_fiscal']) && !empty($row['codigo_postal_fiscal'])
                ];

                sendResponse(true, "Datos fiscales obtenidos", $datosFiscales);
            } else {
                sendResponse(false, "Cliente no encontrado");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    case 'actualizarDatosFiscales':
        $cliente_id = intval($datapost['cliente_id'] ?? 0);
        $rfc = strtoupper(trim($datapost['rfc'] ?? ''));
        $razon_social = trim($datapost['razon_social'] ?? '');
        $regimen_fiscal = trim($datapost['regimen_fiscal'] ?? '');
        $uso_cfdi = trim($datapost['uso_cfdi'] ?? '');
        $codigo_postal_fiscal = trim($datapost['codigo_postal_fiscal'] ?? '');
        $email_facturacion = trim($datapost['email_facturacion'] ?? '');
        $requiere_factura = isset($datapost['requiere_factura']) ? intval($datapost['requiere_factura']) : 0;

        if ($cliente_id <= 0) {
            sendResponse(false, "ID de cliente requerido");
        }

        // Validar RFC
        if (!empty($rfc)) {
            $rfcLength = strlen($rfc);
            if ($rfcLength !== 12 && $rfcLength !== 13) {
                sendResponse(false, "RFC inválido. Debe tener 12 caracteres (moral) o 13 caracteres (física)");
            }
        }

        // Validar código postal
        if (!empty($codigo_postal_fiscal) && !preg_match('/^\d{5}$/', $codigo_postal_fiscal)) {
            sendResponse(false, "Código postal inválido. Debe tener 5 dígitos");
        }

        // Validar email
        if (!empty($email_facturacion) && !filter_var($email_facturacion, FILTER_VALIDATE_EMAIL)) {
            sendResponse(false, "Email de facturación inválido");
        }

        try {
            $query = "UPDATE clientes SET
                        rfc = ?,
                        razon_social = ?,
                        regimen_fiscal = ?,
                        uso_cfdi = ?,
                        codigo_postal_fiscal = ?,
                        email_facturacion = ?,
                        requiere_factura = ?
                      WHERE id = ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("ssssssii", $rfc, $razon_social, $regimen_fiscal, $uso_cfdi,
                              $codigo_postal_fiscal, $email_facturacion, $requiere_factura, $cliente_id);

            if ($stmt->execute()) {
                sendResponse(true, "Datos fiscales actualizados");
            } else {
                sendResponse(false, "Error al actualizar: " . $stmt->error);
            }
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // VENTAS FACTURABLES
    // =====================================================

    case 'getVentasFacturables':
        $cliente_id = intval($datapost['cliente_id'] ?? 0);
        $fecha_inicio = $datapost['fecha_inicio'] ?? null;
        $fecha_fin = $datapost['fecha_fin'] ?? null;

        if ($cliente_id <= 0) {
            sendResponse(false, "ID de cliente requerido");
        }

        try {
            $query = "SELECT v.id, v.folio, v.subtotal, v.impuesto, v.descuento, v.total,
                             v.fecha_creacion as fecha_venta, v.metodo_pago, v.estado,
                             (SELECT COUNT(*) FROM venta_detalles WHERE venta_id = v.id) as total_productos
                      FROM ventas v
                      WHERE v.cliente_id = ?
                        AND (v.facturado = 0 OR v.facturado IS NULL)
                        AND v.id NOT IN (SELECT venta_id FROM factura_ventas)";

            $params = [$cliente_id];
            $types = "i";

            if ($fecha_inicio) {
                $query .= " AND DATE(v.fecha_creacion) >= ?";
                $params[] = $fecha_inicio;
                $types .= "s";
            }

            if ($fecha_fin) {
                $query .= " AND DATE(v.fecha_creacion) <= ?";
                $params[] = $fecha_fin;
                $types .= "s";
            }

            $query .= " ORDER BY v.fecha_creacion DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param($types, ...$params);
            $stmt->execute();
            $result = $stmt->get_result();

            $ventas = [];
            while ($row = $result->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'fecha_venta' => $row['fecha_venta'],
                    'metodo_pago' => $row['metodo_pago'],
                    'estado' => $row['estado'],
                    'total_productos' => intval($row['total_productos'])
                ];
            }

            sendResponse(true, "Ventas facturables obtenidas", $ventas);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // VENTAS FACTURABLES GENERAL (sin filtro de cliente obligatorio)
    // =====================================================

    case 'getVentasFacturablesGeneral':
        $cliente_id = isset($datapost['cliente_id']) && $datapost['cliente_id'] !== '' ? intval($datapost['cliente_id']) : null;
        $fecha_inicio = $datapost['fecha_inicio'] ?? null;
        $fecha_fin = $datapost['fecha_fin'] ?? null;
        $folio = $datapost['folio'] ?? null;
        $caja_id = isset($datapost['caja_id']) && $datapost['caja_id'] !== '' ? intval($datapost['caja_id']) : null;

        try {
            // Query para ventas normales (pedidos/órdenes)
            $queryVentas = "SELECT
                    v.id,
                    v.folio,
                    v.cliente_id,
                    v.cliente_nombre,
                    v.subtotal,
                    v.impuesto,
                    v.descuento,
                    v.total,
                    v.fecha_creacion as fecha_venta,
                    v.metodo_pago,
                    v.estado,
                    (SELECT COUNT(*) FROM venta_detalles WHERE venta_id = v.id) as total_productos,
                    'venta' as tipo_venta
                FROM ventas v
                WHERE (v.facturado = 0 OR v.facturado IS NULL)
                    AND v.id NOT IN (SELECT venta_id FROM factura_ventas WHERE venta_id IS NOT NULL)";

            // Query para ventas de mostrador
            $queryMostrador = "SELECT
                    vm.id,
                    vm.folio,
                    vm.cliente_id,
                    COALESCE(c.nombre, 'Público en General') as cliente_nombre,
                    vm.subtotal,
                    vm.impuesto,
                    vm.descuento,
                    vm.total,
                    vm.fecha_creacion as fecha_venta,
                    vm.metodo_pago,
                    'completada' as estado,
                    (SELECT COUNT(*) FROM ventas_mostrador_detalles WHERE venta_mostrador_id = vm.id) as total_productos,
                    'mostrador' as tipo_venta
                FROM ventas_mostrador vm
                LEFT JOIN clientes c ON vm.cliente_id = c.id
                WHERE (vm.facturado = 0 OR vm.facturado IS NULL)
                    AND vm.id NOT IN (SELECT venta_mostrador_id FROM factura_ventas_mostrador WHERE venta_mostrador_id IS NOT NULL)";

            // Aplicar filtros a ambas queries
            $whereVentas = "";
            $whereMostrador = "";

            if ($cliente_id) {
                $whereVentas .= " AND v.cliente_id = " . intval($cliente_id);
                $whereMostrador .= " AND vm.cliente_id = " . intval($cliente_id);
            }

            if ($fecha_inicio) {
                $whereVentas .= " AND DATE(v.fecha_creacion) >= '" . $conn->real_escape_string($fecha_inicio) . "'";
                $whereMostrador .= " AND DATE(vm.fecha_creacion) >= '" . $conn->real_escape_string($fecha_inicio) . "'";
            }

            if ($fecha_fin) {
                $whereVentas .= " AND DATE(v.fecha_creacion) <= '" . $conn->real_escape_string($fecha_fin) . "'";
                $whereMostrador .= " AND DATE(vm.fecha_creacion) <= '" . $conn->real_escape_string($fecha_fin) . "'";
            }

            if ($folio) {
                $folioEscaped = $conn->real_escape_string($folio);
                $whereVentas .= " AND v.folio LIKE '%" . $folioEscaped . "%'";
                $whereMostrador .= " AND vm.folio LIKE '%" . $folioEscaped . "%'";
            }

            // Filtro de caja solo aplica a ventas de mostrador
            if ($caja_id) {
                $whereMostrador .= " AND vm.caja_id = " . intval($caja_id);
            }

            // Combinar queries con UNION (si hay filtro de caja, excluir ventas normales)
            if ($caja_id) {
                // Solo mostrar ventas de mostrador cuando se filtra por caja
                $query = "(" . $queryMostrador . $whereMostrador . ")
                          ORDER BY fecha_venta DESC LIMIT 100";
            } else {
                // Mostrar ambos tipos de venta
                $query = "(" . $queryVentas . $whereVentas . ")
                          UNION ALL
                          (" . $queryMostrador . $whereMostrador . ")
                          ORDER BY fecha_venta DESC LIMIT 100";
            }

            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error en consulta: " . $conn->error);
            }

            $ventas = [];
            while ($row = $result->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'cliente_id' => $row['cliente_id'] ? intval($row['cliente_id']) : null,
                    'cliente_nombre' => $row['cliente_nombre'] ?: 'Público en General',
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'fecha_venta' => $row['fecha_venta'],
                    'metodo_pago' => $row['metodo_pago'],
                    'estado' => $row['estado'],
                    'total_productos' => intval($row['total_productos']),
                    'tipo_venta' => $row['tipo_venta'] // 'venta' o 'mostrador'
                ];
            }

            sendResponse(true, "Ventas facturables obtenidas", $ventas);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // CONFIGURACIÓN DEL EMISOR
    // =====================================================

    case 'getConfiguracionEmisor':
        try {
            $query = "SELECT clave, valor FROM configuraciones WHERE categoria = 'facturacion'";
            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error al obtener configuración: " . $conn->error);
            }

            $config = [];
            while ($row = $result->fetch_assoc()) {
                $config[$row['clave']] = $row['valor'];
            }

            sendResponse(true, "Configuración del emisor obtenida", $config);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    case 'actualizarConfiguracionEmisor':
        $emisor_rfc = strtoupper(trim($datapost['emisor_rfc'] ?? ''));
        $emisor_razon_social = trim($datapost['emisor_razon_social'] ?? '');
        $emisor_regimen_fiscal = trim($datapost['emisor_regimen_fiscal'] ?? '');
        $emisor_codigo_postal = trim($datapost['emisor_codigo_postal'] ?? '');
        $emisor_lugar_expedicion = trim($datapost['emisor_lugar_expedicion'] ?? '');

        try {
            $conn->begin_transaction();

            $configs = [
                'emisor_rfc' => $emisor_rfc,
                'emisor_razon_social' => $emisor_razon_social,
                'emisor_regimen_fiscal' => $emisor_regimen_fiscal,
                'emisor_codigo_postal' => $emisor_codigo_postal,
                'emisor_lugar_expedicion' => $emisor_lugar_expedicion
            ];

            foreach ($configs as $clave => $valor) {
                $updateQuery = "UPDATE configuraciones SET valor = ? WHERE clave = ?";
                $updateStmt = $conn->prepare($updateQuery);
                $updateStmt->bind_param("ss", $valor, $clave);
                $updateStmt->execute();
            }

            $conn->commit();
            sendResponse(true, "Configuración del emisor actualizada");
        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // CREAR FACTURA
    // =====================================================

    case 'crearFactura':
        $cliente_id = intval($datapost['cliente_id'] ?? 0);
        $venta_ids = $datapost['venta_ids'] ?? [];
        $pago_cliente_id = intval($datapost['pago_cliente_id'] ?? 0);
        $tipo_factura = $datapost['tipo'] ?? 'venta'; // 'venta', 'mostrador' o 'pago'
        $uso_cfdi = trim($datapost['uso_cfdi'] ?? '');
        $forma_pago = trim($datapost['forma_pago'] ?? '');
        $metodo_pago = trim($datapost['metodo_pago'] ?? 'PUE');
        $usuario_id = intval($datapost['usuario_id'] ?? 0);

        // Validaciones
        if ($cliente_id <= 0) {
            sendResponse(false, "Cliente requerido");
        }

        // Validar segun tipo de factura
        if ($tipo_factura === 'pago') {
            if ($pago_cliente_id <= 0) {
                sendResponse(false, "Debe seleccionar un pago para facturar");
            }
        } elseif ($tipo_factura === 'mostrador' || $tipo_factura === 'venta') {
            if (empty($venta_ids) || !is_array($venta_ids)) {
                sendResponse(false, "Debe seleccionar al menos una venta");
            }
        }

        if (empty($uso_cfdi)) {
            sendResponse(false, "Uso de CFDI requerido");
        }

        if (empty($forma_pago)) {
            sendResponse(false, "Forma de pago requerida");
        }

        try {
            $conn->begin_transaction();

            // Obtener datos fiscales del cliente
            $clienteQuery = "SELECT nombre, rfc, razon_social, regimen_fiscal, codigo_postal_fiscal, email_facturacion
                            FROM clientes WHERE id = ?";
            $clienteStmt = $conn->prepare($clienteQuery);
            $clienteStmt->bind_param("i", $cliente_id);
            $clienteStmt->execute();
            $clienteResult = $clienteStmt->get_result();
            $cliente = $clienteResult->fetch_assoc();

            if (!$cliente) {
                $conn->rollback();
                sendResponse(false, "Cliente no encontrado");
            }

            // Validar datos fiscales completos
            if (empty($cliente['rfc']) || empty($cliente['razon_social']) ||
                empty($cliente['regimen_fiscal']) || empty($cliente['codigo_postal_fiscal'])) {
                $conn->rollback();
                sendResponse(false, "El cliente no tiene datos fiscales completos");
            }

            // Obtener configuración del emisor
            $emisorQuery = "SELECT clave, valor FROM configuraciones WHERE categoria = 'facturacion'";
            $emisorResult = $conn->query($emisorQuery);
            $emisor = [];
            while ($row = $emisorResult->fetch_assoc()) {
                $emisor[$row['clave']] = $row['valor'];
            }

            if (empty($emisor['emisor_rfc']) || empty($emisor['emisor_razon_social'])) {
                $conn->rollback();
                sendResponse(false, "Configuración del emisor incompleta");
            }

            // Calcular totales segun tipo de factura
            $pago_info = null;
            $ventaIdsStr = '';

            if ($tipo_factura === 'pago') {
                // Obtener datos del pago
                $pagoQuery = "SELECT pc.*, c.nombre as cliente_nombre
                              FROM pagos_clientes pc
                              JOIN clientes c ON pc.cliente_id = c.id
                              WHERE pc.id = ? AND pc.cliente_id = ?";
                $pagoStmt = $conn->prepare($pagoQuery);
                $pagoStmt->bind_param("ii", $pago_cliente_id, $cliente_id);
                $pagoStmt->execute();
                $pagoResult = $pagoStmt->get_result();
                $pago_info = $pagoResult->fetch_assoc();

                if (!$pago_info) {
                    $conn->rollback();
                    sendResponse(false, "Pago no encontrado o no pertenece al cliente");
                }

                if ($pago_info['facturado'] == 1) {
                    $conn->rollback();
                    sendResponse(false, "Este pago ya fue facturado");
                }

                // Calcular totales desde el pago
                $monto_total = floatval($pago_info['monto_aplicado']);
                $totales = [
                    'subtotal' => round($monto_total / 1.16, 2),
                    'impuesto' => round($monto_total - ($monto_total / 1.16), 2),
                    'descuento' => 0,
                    'total' => $monto_total
                ];
            } elseif ($tipo_factura === 'mostrador') {
                // Calcular totales de las ventas de mostrador
                $ventaIdsStr = implode(',', array_map('intval', $venta_ids));
                $totalesQuery = "SELECT SUM(subtotal) as subtotal, SUM(impuesto) as impuesto,
                                        SUM(descuento) as descuento, SUM(total) as total
                                FROM ventas_mostrador WHERE id IN ($ventaIdsStr)";
                $totalesResult = $conn->query($totalesQuery);
                $totales = $totalesResult->fetch_assoc();
            } else {
                // Calcular totales de las ventas normales (pedidos)
                $ventaIdsStr = implode(',', array_map('intval', $venta_ids));
                $totalesQuery = "SELECT SUM(subtotal) as subtotal, SUM(impuesto) as impuesto,
                                        SUM(descuento) as descuento, SUM(total) as total
                                FROM ventas WHERE id IN ($ventaIdsStr)";
                $totalesResult = $conn->query($totalesQuery);
                $totales = $totalesResult->fetch_assoc();
            }

            // Generar folio
            $folioQuery = "SELECT valor FROM configuraciones WHERE clave = 'facturacion_folio_actual'";
            $folioResult = $conn->query($folioQuery);
            $folioActual = intval($folioResult->fetch_assoc()['valor'] ?? 0) + 1;

            $serieQuery = "SELECT valor FROM configuraciones WHERE clave = 'facturacion_serie_default'";
            $serieResult = $conn->query($serieQuery);
            $serie = $serieResult->fetch_assoc()['valor'] ?? 'A';

            $folio = str_pad($folioActual, 6, '0', STR_PAD_LEFT);

            // Insertar factura
            $insertFacturaQuery = "INSERT INTO facturas (
                folio, serie, cliente_id, usuario_id,
                receptor_rfc, receptor_razon_social, receptor_regimen_fiscal,
                receptor_uso_cfdi, receptor_codigo_postal, receptor_email,
                emisor_rfc, emisor_razon_social, emisor_regimen_fiscal, emisor_codigo_postal,
                subtotal, descuento, total_impuestos, total,
                forma_pago, metodo_pago, moneda, fecha_emision, estado
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'MXN', NOW(), 'pendiente_timbrado')";

            $insertStmt = $conn->prepare($insertFacturaQuery);
            $insertStmt->bind_param(
                "ssiissssssssssddddss",
                $folio, $serie, $cliente_id, $usuario_id,
                $cliente['rfc'], $cliente['razon_social'], $cliente['regimen_fiscal'],
                $uso_cfdi, $cliente['codigo_postal_fiscal'], $cliente['email_facturacion'],
                $emisor['emisor_rfc'], $emisor['emisor_razon_social'],
                $emisor['emisor_regimen_fiscal'], $emisor['emisor_codigo_postal'],
                $totales['subtotal'], $totales['descuento'], $totales['impuesto'], $totales['total'],
                $forma_pago, $metodo_pago
            );

            if (!$insertStmt->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al crear factura: " . $insertStmt->error);
            }

            $factura_id = $conn->insert_id;

            if ($tipo_factura === 'pago') {
                // Para facturación de pago, crear un solo concepto
                $conceptoQuery = "INSERT INTO factura_conceptos (
                    factura_id, producto_id, no_identificacion, descripcion,
                    cantidad, unidad, valor_unitario, importe, descuento,
                    clave_prod_serv, clave_unidad, objeto_imp, base_iva, tasa_iva, importe_iva
                ) VALUES (?, 0, ?, ?, 1, 'Servicio', ?, ?, 0, '84111506', 'E48', '02', ?, 0.160000, ?)";
                $conceptoStmt = $conn->prepare($conceptoQuery);

                $descripcion = "Pago de cuentas por cobrar - Folio " . $pago_info['folio'];
                $no_identificacion = $pago_info['folio'];
                $subtotal = $totales['subtotal'];
                $impuesto = $totales['impuesto'];

                $conceptoStmt->bind_param(
                    "issdddd",
                    $factura_id,
                    $no_identificacion,
                    $descripcion,
                    $subtotal,
                    $subtotal,
                    $subtotal,
                    $impuesto
                );

                if (!$conceptoStmt->execute()) {
                    $conn->rollback();
                    sendResponse(false, "Error al insertar concepto: " . $conceptoStmt->error);
                }

                // Marcar el pago como facturado
                $updatePagoQuery = "UPDATE pagos_clientes SET facturado = 1, factura_id = ?, fecha_facturacion = NOW() WHERE id = ?";
                $updatePagoStmt = $conn->prepare($updatePagoQuery);
                $updatePagoStmt->bind_param("ii", $factura_id, $pago_cliente_id);
                $updatePagoStmt->execute();

            } elseif ($tipo_factura === 'mostrador') {
                // Relacionar ventas de mostrador con la factura
                $relacionQuery = "INSERT INTO factura_ventas_mostrador (factura_id, venta_mostrador_id) VALUES (?, ?)";
                $relacionStmt = $conn->prepare($relacionQuery);

                foreach ($venta_ids as $venta_id) {
                    $venta_id = intval($venta_id);
                    $relacionStmt->bind_param("ii", $factura_id, $venta_id);
                    if (!$relacionStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al relacionar venta de mostrador: " . $relacionStmt->error);
                    }
                }

                // Obtener detalles de las ventas de mostrador e insertar conceptos
                $detallesQuery = "SELECT vmd.producto_id, vmd.producto_nombre as nombre, p.codigo, vmd.cantidad,
                                         vmd.precio_unitario, vmd.subtotal
                                  FROM ventas_mostrador_detalles vmd
                                  JOIN productos p ON vmd.producto_id = p.id
                                  WHERE vmd.venta_mostrador_id IN ($ventaIdsStr)";
                $detallesResult = $conn->query($detallesQuery);

                $conceptoQuery = "INSERT INTO factura_conceptos (
                    factura_id, producto_id, no_identificacion, descripcion,
                    cantidad, unidad, valor_unitario, importe, descuento,
                    clave_prod_serv, clave_unidad, objeto_imp, base_iva, tasa_iva, importe_iva
                ) VALUES (?, ?, ?, ?, ?, 'Pieza', ?, ?, 0, '01010101', 'H87', '02', ?, 0.160000, ?)";
                $conceptoStmt = $conn->prepare($conceptoQuery);

                while ($detalle = $detallesResult->fetch_assoc()) {
                    $base_iva = floatval($detalle['subtotal']);
                    $importe_iva = round($base_iva * 0.16, 2);

                    $conceptoStmt->bind_param(
                        "iissddddd",
                        $factura_id,
                        $detalle['producto_id'],
                        $detalle['codigo'],
                        $detalle['nombre'],
                        $detalle['cantidad'],
                        $detalle['precio_unitario'],
                        $detalle['subtotal'],
                        $base_iva,
                        $importe_iva
                    );

                    if (!$conceptoStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al insertar concepto: " . $conceptoStmt->error);
                    }
                }

                // Marcar ventas de mostrador como facturadas
                $updateVentasQuery = "UPDATE ventas_mostrador SET facturado = 1 WHERE id IN ($ventaIdsStr)";
                $conn->query($updateVentasQuery);

            } else {
                // Relacionar ventas normales (pedidos) con la factura
                $relacionQuery = "INSERT INTO factura_ventas (factura_id, venta_id) VALUES (?, ?)";
                $relacionStmt = $conn->prepare($relacionQuery);

                foreach ($venta_ids as $venta_id) {
                    $venta_id = intval($venta_id);
                    $relacionStmt->bind_param("ii", $factura_id, $venta_id);
                    if (!$relacionStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al relacionar venta: " . $relacionStmt->error);
                    }
                }

                // Obtener detalles de las ventas e insertar conceptos
                $detallesQuery = "SELECT vd.producto_id, p.nombre, p.codigo, vd.cantidad,
                                         vd.precio_unitario, vd.subtotal
                                  FROM venta_detalles vd
                                  JOIN productos p ON vd.producto_id = p.id
                                  WHERE vd.venta_id IN ($ventaIdsStr)";
                $detallesResult = $conn->query($detallesQuery);

                $conceptoQuery = "INSERT INTO factura_conceptos (
                    factura_id, producto_id, no_identificacion, descripcion,
                    cantidad, unidad, valor_unitario, importe, descuento,
                    clave_prod_serv, clave_unidad, objeto_imp, base_iva, tasa_iva, importe_iva
                ) VALUES (?, ?, ?, ?, ?, 'Pieza', ?, ?, 0, '01010101', 'H87', '02', ?, 0.160000, ?)";
                $conceptoStmt = $conn->prepare($conceptoQuery);

                while ($detalle = $detallesResult->fetch_assoc()) {
                    $base_iva = floatval($detalle['subtotal']);
                    $importe_iva = round($base_iva * 0.16, 2);

                    $conceptoStmt->bind_param(
                        "iissddddd",
                        $factura_id,
                        $detalle['producto_id'],
                        $detalle['codigo'],
                        $detalle['nombre'],
                        $detalle['cantidad'],
                        $detalle['precio_unitario'],
                        $detalle['subtotal'],
                        $base_iva,
                        $importe_iva
                    );

                    if (!$conceptoStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al insertar concepto: " . $conceptoStmt->error);
                    }
                }

                // Marcar ventas como facturadas
                $updateVentasQuery = "UPDATE ventas SET facturado = 1 WHERE id IN ($ventaIdsStr)";
                $conn->query($updateVentasQuery);
            }

            // Actualizar folio
            $updateFolioQuery = "UPDATE configuraciones SET valor = ? WHERE clave = 'facturacion_folio_actual'";
            $updateFolioStmt = $conn->prepare($updateFolioQuery);
            $folioStr = strval($folioActual);
            $updateFolioStmt->bind_param("s", $folioStr);
            $updateFolioStmt->execute();

            $conn->commit();

            sendResponse(true, "Factura creada exitosamente", [
                'factura_id' => $factura_id,
                'folio' => $serie . $folio,
                'total' => floatval($totales['total']),
                'estado' => 'pendiente_timbrado'
            ]);

        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al crear factura: " . $e->getMessage());
        }
        break;

    // =====================================================
    // LISTAR FACTURAS
    // =====================================================

    case 'getFacturas':
        $fecha_inicio = $datapost['fecha_inicio'] ?? null;
        $fecha_fin = $datapost['fecha_fin'] ?? null;
        $cliente_id = intval($datapost['cliente_id'] ?? 0);
        $estado = $datapost['estado'] ?? null;
        $folio = trim($datapost['folio'] ?? '');

        try {
            $query = "SELECT f.id, f.uuid, f.folio, f.serie, f.cliente_id,
                             f.receptor_rfc, f.receptor_razon_social,
                             f.subtotal, f.descuento, f.total_impuestos, f.total,
                             f.forma_pago, f.metodo_pago, f.estado,
                             f.fecha_emision, f.fecha_timbrado,
                             c.nombre as cliente_nombre
                      FROM facturas f
                      LEFT JOIN clientes c ON f.cliente_id = c.id
                      WHERE 1=1";

            $params = [];
            $types = "";

            if ($fecha_inicio) {
                $query .= " AND DATE(f.fecha_emision) >= ?";
                $params[] = $fecha_inicio;
                $types .= "s";
            }

            if ($fecha_fin) {
                $query .= " AND DATE(f.fecha_emision) <= ?";
                $params[] = $fecha_fin;
                $types .= "s";
            }

            if ($cliente_id > 0) {
                $query .= " AND f.cliente_id = ?";
                $params[] = $cliente_id;
                $types .= "i";
            }

            if ($estado) {
                $query .= " AND f.estado = ?";
                $params[] = $estado;
                $types .= "s";
            }

            if (!empty($folio)) {
                $query .= " AND (f.folio LIKE ? OR CONCAT(f.serie, f.folio) LIKE ?)";
                $folioSearch = "%$folio%";
                $params[] = $folioSearch;
                $params[] = $folioSearch;
                $types .= "ss";
            }

            $query .= " ORDER BY f.fecha_emision DESC LIMIT 500";

            if (!empty($params)) {
                $stmt = $conn->prepare($query);
                $stmt->bind_param($types, ...$params);
                $stmt->execute();
                $result = $stmt->get_result();
            } else {
                $result = $conn->query($query);
            }

            $facturas = [];
            while ($row = $result->fetch_assoc()) {
                $facturas[] = [
                    'id' => intval($row['id']),
                    'uuid' => $row['uuid'],
                    'folio' => $row['serie'] . $row['folio'],
                    'serie' => $row['serie'],
                    'folio_numero' => $row['folio'],
                    'cliente_id' => intval($row['cliente_id']),
                    'cliente_nombre' => $row['cliente_nombre'],
                    'receptor_rfc' => $row['receptor_rfc'],
                    'receptor_razon_social' => $row['receptor_razon_social'],
                    'subtotal' => floatval($row['subtotal']),
                    'descuento' => floatval($row['descuento']),
                    'total_impuestos' => floatval($row['total_impuestos']),
                    'total' => floatval($row['total']),
                    'forma_pago' => $row['forma_pago'],
                    'metodo_pago' => $row['metodo_pago'],
                    'estado' => $row['estado'],
                    'fecha_emision' => $row['fecha_emision'],
                    'fecha_timbrado' => $row['fecha_timbrado']
                ];
            }

            sendResponse(true, "Facturas obtenidas", $facturas);
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // DETALLE DE FACTURA
    // =====================================================

    case 'getFactura':
        $factura_id = intval($datapost['id'] ?? 0);
        $uuid = trim($datapost['uuid'] ?? '');

        if ($factura_id <= 0 && empty($uuid)) {
            sendResponse(false, "ID o UUID de factura requerido");
        }

        try {
            $query = "SELECT f.*, c.nombre as cliente_nombre
                      FROM facturas f
                      LEFT JOIN clientes c ON f.cliente_id = c.id
                      WHERE " . ($factura_id > 0 ? "f.id = ?" : "f.uuid = ?");

            $stmt = $conn->prepare($query);
            if ($factura_id > 0) {
                $stmt->bind_param("i", $factura_id);
            } else {
                $stmt->bind_param("s", $uuid);
            }
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                $factura_id = intval($row['id']);

                // Obtener conceptos
                $conceptosQuery = "SELECT * FROM factura_conceptos WHERE factura_id = ?";
                $conceptosStmt = $conn->prepare($conceptosQuery);
                $conceptosStmt->bind_param("i", $factura_id);
                $conceptosStmt->execute();
                $conceptosResult = $conceptosStmt->get_result();

                $conceptos = [];
                while ($concepto = $conceptosResult->fetch_assoc()) {
                    $conceptos[] = [
                        'id' => intval($concepto['id']),
                        'producto_id' => intval($concepto['producto_id']),
                        'descripcion' => $concepto['descripcion'],
                        'no_identificacion' => $concepto['no_identificacion'],
                        'cantidad' => floatval($concepto['cantidad']),
                        'unidad' => $concepto['unidad'],
                        'valor_unitario' => floatval($concepto['valor_unitario']),
                        'importe' => floatval($concepto['importe']),
                        'descuento' => floatval($concepto['descuento']),
                        'base_iva' => floatval($concepto['base_iva']),
                        'tasa_iva' => floatval($concepto['tasa_iva']),
                        'importe_iva' => floatval($concepto['importe_iva'])
                    ];
                }

                // Obtener ventas relacionadas
                $ventasQuery = "SELECT v.id, v.folio, v.total, v.fecha_creacion as fecha_venta
                               FROM factura_ventas fv
                               JOIN ventas v ON fv.venta_id = v.id
                               WHERE fv.factura_id = ?";
                $ventasStmt = $conn->prepare($ventasQuery);
                $ventasStmt->bind_param("i", $factura_id);
                $ventasStmt->execute();
                $ventasResult = $ventasStmt->get_result();

                $ventas = [];
                while ($venta = $ventasResult->fetch_assoc()) {
                    $ventas[] = [
                        'id' => intval($venta['id']),
                        'folio' => $venta['folio'],
                        'total' => floatval($venta['total']),
                        'fecha_venta' => $venta['fecha_venta']
                    ];
                }

                $factura = [
                    'id' => intval($row['id']),
                    'uuid' => $row['uuid'],
                    'folio' => $row['serie'] . $row['folio'],
                    'serie' => $row['serie'],
                    'folio_numero' => $row['folio'],
                    'cliente_id' => intval($row['cliente_id']),
                    'cliente_nombre' => $row['cliente_nombre'],
                    'receptor' => [
                        'rfc' => $row['receptor_rfc'],
                        'razon_social' => $row['receptor_razon_social'],
                        'regimen_fiscal' => $row['receptor_regimen_fiscal'],
                        'uso_cfdi' => $row['receptor_uso_cfdi'],
                        'codigo_postal' => $row['receptor_codigo_postal'],
                        'email' => $row['receptor_email']
                    ],
                    'emisor' => [
                        'rfc' => $row['emisor_rfc'],
                        'razon_social' => $row['emisor_razon_social'],
                        'regimen_fiscal' => $row['emisor_regimen_fiscal'],
                        'codigo_postal' => $row['emisor_codigo_postal']
                    ],
                    'subtotal' => floatval($row['subtotal']),
                    'descuento' => floatval($row['descuento']),
                    'total_impuestos' => floatval($row['total_impuestos']),
                    'total' => floatval($row['total']),
                    'forma_pago' => $row['forma_pago'],
                    'metodo_pago' => $row['metodo_pago'],
                    'moneda' => $row['moneda'],
                    'estado' => $row['estado'],
                    'ruta_xml' => $row['ruta_xml'],
                    'ruta_pdf' => $row['ruta_pdf'],
                    'fecha_emision' => $row['fecha_emision'],
                    'fecha_timbrado' => $row['fecha_timbrado'],
                    'fecha_cancelacion' => $row['fecha_cancelacion'],
                    'motivo_cancelacion' => $row['motivo_cancelacion'],
                    'conceptos' => $conceptos,
                    'ventas' => $ventas
                ];

                sendResponse(true, "Factura obtenida", $factura);
            } else {
                sendResponse(false, "Factura no encontrada");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // DESCARGAR XML/PDF
    // =====================================================

    case 'descargarFacturaXML':
        $factura_id = intval($datapost['factura_id'] ?? 0);

        if ($factura_id <= 0) {
            sendResponse(false, "ID de factura requerido");
        }

        try {
            $query = "SELECT ruta_xml, folio, serie FROM facturas WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $factura_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                if (!empty($row['ruta_xml'])) {
                    sendResponse(true, "URL de XML", [
                        'url' => $row['ruta_xml'],
                        'filename' => $row['serie'] . $row['folio'] . '.xml'
                    ]);
                } else {
                    sendResponse(false, "El XML aún no está disponible (factura pendiente de timbrado)");
                }
            } else {
                sendResponse(false, "Factura no encontrada");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    case 'descargarFacturaPDF':
        $factura_id = intval($datapost['factura_id'] ?? 0);

        if ($factura_id <= 0) {
            sendResponse(false, "ID de factura requerido");
        }

        try {
            $query = "SELECT ruta_pdf, folio, serie FROM facturas WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $factura_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                if (!empty($row['ruta_pdf'])) {
                    sendResponse(true, "URL de PDF", [
                        'url' => $row['ruta_pdf'],
                        'filename' => $row['serie'] . $row['folio'] . '.pdf'
                    ]);
                } else {
                    sendResponse(false, "El PDF aún no está disponible (factura pendiente de timbrado)");
                }
            } else {
                sendResponse(false, "Factura no encontrada");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    // =====================================================
    // REENVIAR FACTURA POR EMAIL
    // =====================================================

    case 'reenviarFactura':
        $factura_id = intval($datapost['factura_id'] ?? 0);
        $email = trim($datapost['email'] ?? '');

        if ($factura_id <= 0) {
            sendResponse(false, "ID de factura requerido");
        }

        try {
            // Obtener datos de la factura
            $query = "SELECT receptor_email, ruta_xml, ruta_pdf, folio, serie, estado
                      FROM facturas WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $factura_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                if ($row['estado'] === 'pendiente_timbrado') {
                    sendResponse(false, "La factura aún no ha sido timbrada");
                }

                $emailDestino = !empty($email) ? $email : $row['receptor_email'];

                if (empty($emailDestino)) {
                    sendResponse(false, "No hay email de destino");
                }

                // TODO: Implementar envío de email real
                // Por ahora solo simular
                sendResponse(true, "Factura enviada a " . $emailDestino, [
                    'email' => $emailDestino,
                    'folio' => $row['serie'] . $row['folio']
                ]);
            } else {
                sendResponse(false, "Factura no encontrada");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para facturación: " . $tipo);
        break;
}
?>
