<?php
/**
 * IngresosEfectivo.php
 * Endpoints para gestionar ingresos de efectivo a caja (fondos, cambio, etc.)
 */

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

switch ($tipo) {
    case 'verificarPermisoIngreso':
        ing_verificarPermisoIngreso();
        break;
    case 'crearIngresoEfectivo':
        ing_crearIngresoEfectivo();
        break;
    case 'getIngresosEfectivo':
        ing_getIngresosEfectivo();
        break;
    case 'getIngresosPendientesCorte':
        ing_getIngresosPendientesCorte();
        break;
    default:
        sendResponse(false, "Operación de ingresos no implementada: " . $tipo);
        break;
}

/**
 * Verificar contraseña y permiso para ingreso de efectivo
 */
function ing_verificarPermisoIngreso() {
    global $conn, $datapost;

    $password = isset($datapost['password']) ? $datapost['password'] : null;

    if (!$password) {
        sendResponse(false, "Contraseña requerida");
        return;
    }

    try {
        // Buscar todos los usuarios activos
        $stmtUsers = $conn->prepare("SELECT id, password, nombre, rol_id FROM usuarios WHERE activo = 1");
        $stmtUsers->execute();
        $usersResult = $stmtUsers->get_result();

        $usuarioEncontrado = null;

        // Buscar qué usuario tiene esta contraseña
        while ($user = $usersResult->fetch_assoc()) {
            if (password_verify($password, $user['password'])) {
                $usuarioEncontrado = $user;
                break;
            }
        }

        if (!$usuarioEncontrado) {
            sendResponse(false, "Contraseña incorrecta");
            return;
        }

        // Verificar permiso de ingreso de efectivo
        $stmtPermiso = $conn->prepare("
            SELECT 1 FROM rol_permisos rp
            INNER JOIN permisos p ON rp.permiso_id = p.id
            WHERE rp.rol_id = ? AND p.nombre = 'ingreso_efectivo'
        ");
        $stmtPermiso->bind_param("i", $usuarioEncontrado['rol_id']);
        $stmtPermiso->execute();
        $permisoResult = $stmtPermiso->get_result();

        if ($permisoResult->num_rows === 0) {
            sendResponse(false, "El usuario " . $usuarioEncontrado['nombre'] . " no tiene permiso para registrar ingresos de efectivo");
            return;
        }

        sendResponse(true, "Acceso autorizado", [
            'usuario_id' => $usuarioEncontrado['id'],
            'usuario_nombre' => $usuarioEncontrado['nombre']
        ]);

    } catch (Exception $e) {
        sendResponse(false, "Error al verificar permiso: " . $e->getMessage());
    }
}

/**
 * Crear un ingreso de efectivo
 */
function ing_crearIngresoEfectivo() {
    global $conn, $datapost;

    $usuarioId = isset($datapost['usuario_id']) ? (int)$datapost['usuario_id'] : null;
    $monto = isset($datapost['monto']) ? (float)$datapost['monto'] : 0;
    $concepto = isset($datapost['concepto']) ? trim($datapost['concepto']) : null;
    $descripcion = isset($datapost['descripcion']) ? trim($datapost['descripcion']) : null;
    $nombrePc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : null;

    // Validaciones
    if (!$usuarioId) {
        sendResponse(false, "Usuario requerido");
        return;
    }

    if ($monto <= 0) {
        sendResponse(false, "El monto debe ser mayor a 0");
        return;
    }

    if (!$concepto) {
        sendResponse(false, "El concepto es requerido");
        return;
    }

    if (!$nombrePc) {
        sendResponse(false, "Nombre de PC requerido para identificar la caja");
        return;
    }

    try {
        // Obtener caja_id por nombre_pc
        $stmtCaja = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
        $stmtCaja->bind_param("s", $nombrePc);
        $stmtCaja->execute();
        $cajaResult = $stmtCaja->get_result();

        if ($cajaResult->num_rows === 0) {
            sendResponse(false, "Caja no encontrada para este equipo");
            return;
        }

        $cajaId = $cajaResult->fetch_assoc()['id'];

        // Insertar ingreso
        $stmt = $conn->prepare("
            INSERT INTO ingresos_efectivo (caja_id, usuario_id, monto, concepto, descripcion, nombre_pc, fecha_ingreso, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW(), NOW())
        ");
        $stmt->bind_param("iidsss", $cajaId, $usuarioId, $monto, $concepto, $descripcion, $nombrePc);

        if ($stmt->execute()) {
            $ingresoId = $conn->insert_id;

            sendResponse(true, "Ingreso registrado correctamente", [
                'id' => $ingresoId,
                'folio' => $ingresoId,
                'monto' => $monto,
                'concepto' => $concepto,
                'fecha' => date('Y-m-d H:i:s')
            ]);
        } else {
            throw new Exception("Error al registrar ingreso: " . $conn->error);
        }
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener ingresos de efectivo (con filtros opcionales)
 */
function ing_getIngresosEfectivo() {
    global $conn, $datapost;

    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);
    $fechaInicio = isset($_GET['fecha_inicio']) ? $_GET['fecha_inicio'] : (isset($datapost['fecha_inicio']) ? $datapost['fecha_inicio'] : null);
    $fechaFin = isset($_GET['fecha_fin']) ? $_GET['fecha_fin'] : (isset($datapost['fecha_fin']) ? $datapost['fecha_fin'] : null);

    try {
        $query = "
            SELECT
                i.*,
                u.nombre as usuario_nombre,
                c.identificador as caja_nombre
            FROM ingresos_efectivo i
            LEFT JOIN usuarios u ON i.usuario_id = u.id
            LEFT JOIN cajas c ON i.caja_id = c.id
            WHERE 1=1
        ";
        $params = [];
        $types = "";

        if ($nombrePc) {
            $query .= " AND i.nombre_pc = ?";
            $params[] = $nombrePc;
            $types .= "s";
        }

        if ($fechaInicio) {
            $query .= " AND DATE(i.fecha_ingreso) >= ?";
            $params[] = $fechaInicio;
            $types .= "s";
        }

        if ($fechaFin) {
            $query .= " AND DATE(i.fecha_ingreso) <= ?";
            $params[] = $fechaFin;
            $types .= "s";
        }

        $query .= " ORDER BY i.fecha_ingreso DESC LIMIT 100";

        $stmt = $conn->prepare($query);
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        $result = $stmt->get_result();

        $ingresos = [];
        while ($row = $result->fetch_assoc()) {
            $ingresos[] = $row;
        }

        sendResponse(true, "Ingresos obtenidos", $ingresos);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener total de ingresos desde el último corte de caja
 */
function ing_getIngresosPendientesCorte() {
    global $conn, $datapost;

    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);

    if (!$nombrePc) {
        sendResponse(false, "Nombre de PC requerido");
        return;
    }

    try {
        // Obtener caja_id
        $stmtCaja = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
        $stmtCaja->bind_param("s", $nombrePc);
        $stmtCaja->execute();
        $cajaResult = $stmtCaja->get_result();

        if ($cajaResult->num_rows === 0) {
            sendResponse(false, "Caja no encontrada");
            return;
        }

        $cajaId = $cajaResult->fetch_assoc()['id'];

        // Obtener fecha del último corte
        $stmtCorte = $conn->prepare("
            SELECT fecha_creacion FROM cortes_caja
            WHERE caja_id = ?
            ORDER BY fecha_creacion DESC
            LIMIT 1
        ");
        $stmtCorte->bind_param("i", $cajaId);
        $stmtCorte->execute();
        $corteResult = $stmtCorte->get_result();

        $fechaUltimoCorte = null;
        if ($corteResult->num_rows > 0) {
            $fechaUltimoCorte = $corteResult->fetch_assoc()['fecha_creacion'];
        }

        // Obtener total de ingresos desde el último corte
        if ($fechaUltimoCorte) {
            $stmtIngresos = $conn->prepare("
                SELECT
                    COALESCE(SUM(monto), 0) as total_ingresos,
                    COUNT(*) as cantidad_ingresos
                FROM ingresos_efectivo
                WHERE caja_id = ? AND fecha_ingreso > ?
            ");
            $stmtIngresos->bind_param("is", $cajaId, $fechaUltimoCorte);
        } else {
            $stmtIngresos = $conn->prepare("
                SELECT
                    COALESCE(SUM(monto), 0) as total_ingresos,
                    COUNT(*) as cantidad_ingresos
                FROM ingresos_efectivo
                WHERE caja_id = ?
            ");
            $stmtIngresos->bind_param("i", $cajaId);
        }

        $stmtIngresos->execute();
        $ingresosResult = $stmtIngresos->get_result();
        $totales = $ingresosResult->fetch_assoc();

        sendResponse(true, "Ingresos pendientes de corte", [
            'total_ingresos' => (float)$totales['total_ingresos'],
            'cantidad_ingresos' => (int)$totales['cantidad_ingresos'],
            'ultimo_corte' => $fechaUltimoCorte
        ]);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}
