<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

// Endpoints para Precios Ray (permiso: precios_ray)

switch ($tipo) {
    case 'getProductosParaRay':
        // Endpoint específico para Precios Ray
        // Excluye productos base que tengan equivalencias (muestra solo el equivalente mayor)
        $pagina = intval($datapost['pagina'] ?? 1);
        $limite = intval($datapost['limite'] ?? 50);
        $busqueda = trim($datapost['busqueda'] ?? '');
        $categoria_id = isset($datapost['categoria_id']) && $datapost['categoria_id'] !== '' ? intval($datapost['categoria_id']) : null;

        if ($pagina < 1) $pagina = 1;
        if ($limite < 1) $limite = 50;
        if ($limite > 100) $limite = 100;

        $offset = ($pagina - 1) * $limite;

        try {
            // Condición para excluir productos base que tengan equivalencias activas
            $whereConditions = [
                "p.activo = 1",
                "p.id NOT IN (SELECT pe.producto_equivalente_id FROM producto_equivalencias pe WHERE pe.activo = 1)"
            ];
            $params = [];
            $types = "";

            if (!empty($busqueda)) {
                // Separar palabras para buscar cada una con AND
                $palabras = preg_split('/\s+/', $busqueda);
                $nombreConditions = [];
                foreach ($palabras as $palabra) {
                    $palabra = trim($palabra);
                    if ($palabra === '') continue;
                    // Normalizar: si es número, buscar también con cero adelante y con #
                    if (is_numeric($palabra)) {
                        $padded = str_pad($palabra, 2, '0', STR_PAD_LEFT);
                        $nombreConditions[] = "(p.nombre LIKE ? OR p.nombre LIKE ? OR p.codigo_barras LIKE ? OR p.codigo_corto LIKE ?)";
                        $params[] = "%$palabra%";
                        $params[] = "%$padded%";
                        $params[] = "%$palabra%";
                        $params[] = "%$palabra%";
                        $types .= "ssss";
                    } else {
                        $nombreConditions[] = "(p.nombre LIKE ? OR p.codigo_barras LIKE ? OR p.codigo_corto LIKE ?)";
                        $params[] = "%$palabra%";
                        $params[] = "%$palabra%";
                        $params[] = "%$palabra%";
                        $types .= "sss";
                    }
                }
                if (!empty($nombreConditions)) {
                    $whereConditions[] = "(" . implode(" AND ", $nombreConditions) . ")";
                }
            }

            if ($categoria_id !== null) {
                $whereConditions[] = "p.categoria_id = ?";
                $params[] = $categoria_id;
                $types .= "i";
            }

            $whereClause = implode(" AND ", $whereConditions);

            // Query para contar el total
            $countQuery = "SELECT COUNT(*) as total FROM productos p WHERE $whereClause";
            $stmtCount = $conn->prepare($countQuery);
            if (!empty($params)) {
                $stmtCount->bind_param($types, ...$params);
            }
            $stmtCount->execute();
            $countResult = $stmtCount->get_result();
            $totalRow = $countResult->fetch_assoc();
            $total = intval($totalRow['total']);
            $stmtCount->close();

            // Query para obtener productos
            $query = "SELECT
                p.id,
                p.nombre,
                p.descripcion,
                p.codigo_corto,
                p.codigo_barras,
                p.categoria_id,
                c.nombre as categoria_nombre,
                p.almacen_id,
                a.nombre as almacen_nombre,
                p.costo_precios,
                p.precio_lista,
                p.precio1,
                p.precio2,
                p.precio3,
                p.precio4,
                p.precio5,
                p.precio6,
                p.stock,
                p.stock_minimo,
                p.unidad_medida,
                p.impresora,
                p.activo,
                (SELECT dv.cantidad_minima FROM descuentos_volumen dv WHERE dv.producto_id = p.id AND dv.activo = 1 ORDER BY dv.cantidad_minima ASC LIMIT 1) as cantidad_descuento,
                (SELECT dv.descuento_porcentaje FROM descuentos_volumen dv WHERE dv.producto_id = p.id AND dv.activo = 1 ORDER BY dv.cantidad_minima ASC LIMIT 1) as porcentaje_descuento,
                pe.producto_equivalente_id as producto_base_id,
                pe.cantidad as equivalencia,
                pb.precio_lista as precio_base_lista,
                pb.precio1 as precio_base_precio1,
                (SELECT dv.cantidad_minima FROM descuentos_volumen dv WHERE dv.producto_id = pb.id AND dv.activo = 1 ORDER BY dv.cantidad_minima ASC LIMIT 1) as cantidad_descuento_base,
                (SELECT dv.descuento_porcentaje FROM descuentos_volumen dv WHERE dv.producto_id = pb.id AND dv.activo = 1 ORDER BY dv.cantidad_minima ASC LIMIT 1) as porcentaje_descuento_base
            FROM productos p
            LEFT JOIN categorias c ON p.categoria_id = c.id
            LEFT JOIN almacenes a ON p.almacen_id = a.id
            LEFT JOIN producto_equivalencias pe ON pe.producto_id = p.id AND pe.activo = 1
            LEFT JOIN productos pb ON pb.id = pe.producto_equivalente_id
            WHERE $whereClause
            ORDER BY p.nombre ASC
            LIMIT ? OFFSET ?";

            $params[] = $limite;
            $params[] = $offset;
            $types .= "ii";

            $stmt = $conn->prepare($query);
            $stmt->bind_param($types, ...$params);
            $stmt->execute();
            $result = $stmt->get_result();
            $productos = [];

            while ($row = $result->fetch_assoc()) {
                $productos[] = [
                    'id' => intval($row['id']),
                    'nombre' => $row['nombre'],
                    'descripcion' => $row['descripcion'],
                    'codigo_corto' => $row['codigo_corto'],
                    'codigo_barras' => $row['codigo_barras'],
                    'categoria_id' => $row['categoria_id'] ? intval($row['categoria_id']) : null,
                    'categoria_nombre' => $row['categoria_nombre'],
                    'almacen_id' => $row['almacen_id'] ? intval($row['almacen_id']) : null,
                    'almacen_nombre' => $row['almacen_nombre'],
                    'costo_precios' => floatval($row['costo_precios']),
                    'precio_lista' => floatval($row['precio_lista']),
                    'precio1' => floatval($row['precio1']),
                    'precio2' => floatval($row['precio2']),
                    'precio3' => floatval($row['precio3']),
                    'precio4' => floatval($row['precio4']),
                    'precio5' => floatval($row['precio5']),
                    'precio6' => floatval($row['precio6']),
                    'stock' => intval($row['stock']),
                    'stock_minimo' => intval($row['stock_minimo']),
                    'unidad_medida' => $row['unidad_medida'],
                    'impresora' => $row['impresora'],
                    'activo' => intval($row['activo']),
                    'es_base' => intval($row['es_base'] ?? 1),
                    'cantidad_descuento' => $row['cantidad_descuento'] ? floatval($row['cantidad_descuento']) : 0,
                    'porcentaje_descuento' => $row['porcentaje_descuento'] ? floatval($row['porcentaje_descuento']) : 0,
                    'estado_stock' => getEstadoStock($row['stock'], $row['stock_minimo']),
                    'producto_base_id' => $row['producto_base_id'] ? intval($row['producto_base_id']) : null,
                    'equivalencia' => $row['equivalencia'] ? floatval($row['equivalencia']) : null,
                    'precio_base_lista' => $row['precio_base_lista'] ? floatval($row['precio_base_lista']) : null,
                    'precio_base_precio1' => $row['precio_base_precio1'] ? floatval($row['precio_base_precio1']) : null,
                    'cantidad_descuento_base' => $row['cantidad_descuento_base'] ? floatval($row['cantidad_descuento_base']) : 0,
                    'porcentaje_descuento_base' => $row['porcentaje_descuento_base'] ? floatval($row['porcentaje_descuento_base']) : 0
                ];
            }
            $stmt->close();

            $totalPaginas = ceil($total / $limite);
            $hayMas = $pagina < $totalPaginas;

            sendResponse(true, "Productos obtenidos exitosamente", [
                'productos' => $productos,
                'paginacion' => [
                    'pagina' => $pagina,
                    'limite' => $limite,
                    'total' => $total,
                    'total_paginas' => $totalPaginas,
                    'hay_mas' => $hayMas
                ]
            ]);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener productos: " . $e->getMessage());
        }
        break;

    case 'actualizarPreciosRay':
        $id = intval($datapost['id'] ?? 0);
        $porcentajeBase = isset($datapost['porcentaje_base']) ? floatval($datapost['porcentaje_base']) : 0;

        if ($id <= 0) {
            sendResponse(false, "ID de producto requerido");
        }

        try {
            $conn->begin_transaction();

            // Verificar que el producto existe
            $checkQuery = "SELECT id FROM productos WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $id);
            $checkStmt->execute();

            if ($checkStmt->get_result()->num_rows === 0) {
                $conn->rollback();
                sendResponse(false, "Producto no encontrado");
            }

            // Construir la consulta dinámicamente solo con los campos enviados
            $camposPermitidos = ['costo_precios', 'precio_lista', 'precio1', 'precio2', 'precio3', 'precio4', 'precio5', 'precio6'];
            $preciosActualizados = [];
            $setClauses = [];
            $params = [];
            $types = "";

            foreach ($camposPermitidos as $campo) {
                if (isset($datapost[$campo])) {
                    $valor = floatval($datapost[$campo]);
                    $setClauses[] = "$campo = ?";
                    $params[] = $valor;
                    $types .= "d";
                    $preciosActualizados[$campo] = $valor;
                }
            }

            if (empty($setClauses)) {
                $conn->rollback();
                sendResponse(false, "No se proporcionaron precios para actualizar");
            }

            // Agregar fecha de actualización
            $setClauses[] = "fecha_actualizacion = NOW()";

            // Agregar el ID al final de los parámetros
            $params[] = $id;
            $types .= "i";

            $query = "UPDATE productos SET " . implode(", ", $setClauses) . " WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param($types, ...$params);

            if (!$stmt->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al actualizar precios: " . $stmt->error);
            }

            // Buscar si este producto es equivalente (el grande) de algún producto base
            $equivQuery = "SELECT pe.producto_equivalente_id as producto_base_id, pe.cantidad
                           FROM producto_equivalencias pe
                           WHERE pe.producto_id = ? AND pe.activo = 1";
            $equivStmt = $conn->prepare($equivQuery);
            $equivStmt->bind_param("i", $id);
            $equivStmt->execute();
            $equivResult = $equivStmt->get_result();

            $productosBaseActualizados = 0;
            while ($equivRow = $equivResult->fetch_assoc()) {
                $productoBaseId = intval($equivRow['producto_base_id']);
                $cantidadEquivalencia = floatval($equivRow['cantidad']);

                if ($cantidadEquivalencia > 0) {
                    // Calcular precios del producto base
                    $setClausesBase = [];
                    $paramsBase = [];
                    $typesBase = "";

                    foreach ($preciosActualizados as $campo => $valor) {
                        // Solo precio_lista lleva el %Base de incremento
                        // precio_base_lista = (precio_lista_caja / equivalencia) * (1 + %base/100)
                        if ($campo === 'precio_lista' && $porcentajeBase != 0) {
                            $valorBase = round(($valor / $cantidadEquivalencia) * (1 + $porcentajeBase / 100), 3);
                        } else {
                            // Para los demás precios, solo dividir por equivalencia
                            $valorBase = round($valor / $cantidadEquivalencia, 3);
                        }
                        $setClausesBase[] = "$campo = ?";
                        $paramsBase[] = $valorBase;
                        $typesBase .= "d";
                    }

                    if (!empty($setClausesBase)) {
                        $setClausesBase[] = "fecha_actualizacion = NOW()";
                        $paramsBase[] = $productoBaseId;
                        $typesBase .= "i";

                        $queryBase = "UPDATE productos SET " . implode(", ", $setClausesBase) . " WHERE id = ?";
                        $stmtBase = $conn->prepare($queryBase);
                        $stmtBase->bind_param($typesBase, ...$paramsBase);

                        if ($stmtBase->execute()) {
                            $productosBaseActualizados++;
                        }
                    }
                }
            }

            $conn->commit();

            $mensaje = "Precios actualizados exitosamente";
            if ($productosBaseActualizados > 0) {
                $mensaje .= ". Se actualizaron $productosBaseActualizados producto(s) base por equivalencia.";
            }
            sendResponse(true, $mensaje);

        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al actualizar precios: " . $e->getMessage());
        }
        break;

    case 'updateDescuentoVolumenRay':
        $producto_id = intval($datapost['producto_id'] ?? 0);
        $cantidad_minima = floatval($datapost['cantidad_minima'] ?? 0);
        $descuento_porcentaje = isset($datapost['descuento_porcentaje']) ? floatval($datapost['descuento_porcentaje']) : null;

        if ($producto_id <= 0) {
            sendResponse(false, "ID de producto requerido");
        }

        try {
            // Si cantidad es 0, eliminar descuento
            if ($cantidad_minima <= 0) {
                $deleteQuery = "UPDATE descuentos_volumen SET activo = 0 WHERE producto_id = ?";
                $deleteStmt = $conn->prepare($deleteQuery);
                $deleteStmt->bind_param("i", $producto_id);
                $deleteStmt->execute();
                sendResponse(true, "Descuento eliminado");
            }

            // Verificar si ya existe un descuento para este producto
            $checkQuery = "SELECT id FROM descuentos_volumen WHERE producto_id = ? AND activo = 1";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $producto_id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows > 0) {
                // Actualizar existente
                $row = $checkResult->fetch_assoc();
                $descuento_id = $row['id'];

                if ($descuento_porcentaje !== null) {
                    $updateQuery = "UPDATE descuentos_volumen SET cantidad_minima = ?, descuento_porcentaje = ? WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->bind_param("ddi", $cantidad_minima, $descuento_porcentaje, $descuento_id);
                } else {
                    $updateQuery = "UPDATE descuentos_volumen SET cantidad_minima = ? WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->bind_param("di", $cantidad_minima, $descuento_id);
                }
                $updateStmt->execute();
                sendResponse(true, "Descuento actualizado");
            } else {
                // Crear nuevo
                $porcentaje = $descuento_porcentaje ?? 0;
                $insertQuery = "INSERT INTO descuentos_volumen (producto_id, cantidad_minima, descuento_porcentaje, activo) VALUES (?, ?, ?, 1)";
                $insertStmt = $conn->prepare($insertQuery);
                $insertStmt->bind_param("idd", $producto_id, $cantidad_minima, $porcentaje);
                $insertStmt->execute();
                sendResponse(true, "Descuento creado");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al actualizar descuento: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para Precios Ray");
        break;
}
