<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'getEquivalenciasProducto':
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            $query = "SELECT
                pe.id,
                pe.producto_id,
                pe.producto_equivalente_id,
                pe.cantidad,
                pe.activo,
                p.nombre as producto_equivalente_nombre,
                p.codigo_corto,
                p.codigo_barras,
                p.unidad_medida,
                p.stock,
                p.precio_compra,
                p.precio_lista,
                p.precio1,
                p.precio2,
                p.precio3,
                p.precio4,
                p.precio5,
                p.precio6,
                dv.cantidad_minima as volumen,
                dv.descuento_porcentaje as descuento
            FROM producto_equivalencias pe
            INNER JOIN productos p ON pe.producto_equivalente_id = p.id
            LEFT JOIN descuentos_volumen dv ON dv.producto_id = pe.producto_equivalente_id
            WHERE pe.producto_id = ? AND pe.activo = 1
            ORDER BY p.nombre ASC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $producto_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $equivalencias = [];
            while ($row = $result->fetch_assoc()) {
                $equivalencias[] = [
                    'id' => intval($row['id']),
                    'producto_id' => intval($row['producto_id']),
                    'producto_equivalente_id' => intval($row['producto_equivalente_id']),
                    'producto_equivalente_nombre' => $row['producto_equivalente_nombre'],
                    'codigo_corto' => $row['codigo_corto'],
                    'codigo_barras' => $row['codigo_barras'],
                    'cantidad' => floatval($row['cantidad']),
                    'unidad_medida' => $row['unidad_medida'],
                    'stock' => intval($row['stock']),
                    'costo' => floatval($row['precio_compra'] ?? 0),
                    'precio_lista' => floatval($row['precio_lista'] ?? 0),
                    'precio1' => floatval($row['precio1'] ?? 0),
                    'precio2' => floatval($row['precio2'] ?? 0),
                    'precio3' => floatval($row['precio3'] ?? 0),
                    'precio4' => floatval($row['precio4'] ?? 0),
                    'precio5' => floatval($row['precio5'] ?? 0),
                    'precio6' => floatval($row['precio6'] ?? 0),
                    'volumen' => floatval($row['volumen'] ?? 0),
                    'descuento' => floatval($row['descuento'] ?? 0),
                    'activo' => intval($row['activo'])
                ];
            }

            sendResponse(true, "Equivalencias obtenidas correctamente", $equivalencias);
        } catch (Exception $e) {
            error_log("Error al obtener equivalencias: " . $e->getMessage());
            sendResponse(false, "Error al obtener equivalencias");
        }
        break;

    case 'guardarEquivalenciasProducto':
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;
            $equivalencias = isset($datapost['equivalencias']) ? $datapost['equivalencias'] : [];

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            // Verificar que el producto existe
            $checkQuery = "SELECT id FROM productos WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $producto_id);
            $checkStmt->execute();

            if ($checkStmt->get_result()->num_rows === 0) {
                sendResponse(false, "Producto no encontrado");
                break;
            }

            // Iniciar transacción
            $conn->begin_transaction();

            try {
                // Desactivar todas las equivalencias existentes
                $deleteQuery = "UPDATE producto_equivalencias SET activo = 0 WHERE producto_id = ?";
                $deleteStmt = $conn->prepare($deleteQuery);
                $deleteStmt->bind_param("i", $producto_id);
                $deleteStmt->execute();

                // Insertar o actualizar equivalencias
                $insertQuery = "INSERT INTO producto_equivalencias
                    (producto_id, producto_equivalente_id, cantidad, activo)
                    VALUES (?, ?, ?, 1)
                    ON DUPLICATE KEY UPDATE
                    cantidad = VALUES(cantidad),
                    activo = 1,
                    fecha_actualizacion = CURRENT_TIMESTAMP";

                $insertStmt = $conn->prepare($insertQuery);

                foreach ($equivalencias as $equiv) {
                    $producto_equiv_id = intval($equiv['producto_equivalente_id']);
                    $cantidad = floatval($equiv['cantidad'] ?? 1);
                    $volumen = floatval($equiv['volumen'] ?? 0);
                    $descuento = floatval($equiv['descuento'] ?? 0);

                    // Validaciones
                    if ($producto_equiv_id <= 0) {
                        throw new Exception("ID de producto equivalente inválido");
                    }

                    if ($cantidad <= 0) {
                        $cantidad = 1;
                    }

                    // No permitir que un producto sea equivalente a sí mismo
                    if ($producto_id === $producto_equiv_id) {
                        throw new Exception("Un producto no puede ser equivalente a sí mismo");
                    }

                    // Verificar que el producto equivalente existe
                    $checkEquivQuery = "SELECT id FROM productos WHERE id = ? AND activo = 1";
                    $checkEquivStmt = $conn->prepare($checkEquivQuery);
                    $checkEquivStmt->bind_param("i", $producto_equiv_id);
                    $checkEquivStmt->execute();

                    if ($checkEquivStmt->get_result()->num_rows === 0) {
                        throw new Exception("El producto equivalente no existe o está inactivo");
                    }

                    // Insertar relación de equivalencia
                    $insertStmt->bind_param("iid", $producto_id, $producto_equiv_id, $cantidad);
                    $insertStmt->execute();

                    // Si hay volumen y descuento, crear/actualizar descuento por volumen para el producto equivalente
                    if ($volumen > 0 && $descuento > 0) {
                        // Eliminar descuentos existentes para este producto equivalente
                        $deleteDescQuery = "DELETE FROM descuentos_volumen WHERE producto_id = ?";
                        $deleteDescStmt = $conn->prepare($deleteDescQuery);
                        $deleteDescStmt->bind_param("i", $producto_equiv_id);
                        $deleteDescStmt->execute();

                        // Insertar nuevo descuento por volumen
                        $insertDescQuery = "INSERT INTO descuentos_volumen
                            (producto_id, cantidad_minima, descuento_porcentaje)
                            VALUES (?, ?, ?)";
                        $insertDescStmt = $conn->prepare($insertDescQuery);
                        $insertDescStmt->bind_param("idd", $producto_equiv_id, $volumen, $descuento);
                        $insertDescStmt->execute();
                    }

                    // Actualizar precios del producto equivalente si se proporcionan
                    if (isset($equiv['precio_lista']) || isset($equiv['costo'])) {
                        $updateFields = [];
                        $updateValues = [];
                        $updateTypes = "";

                        if (isset($equiv['codigo_corto'])) {
                            $updateFields[] = "codigo_corto = ?";
                            $updateValues[] = $equiv['codigo_corto'];
                            $updateTypes .= "s";
                        }
                        if (isset($equiv['codigo_barras'])) {
                            $updateFields[] = "codigo_barras = ?";
                            $updateValues[] = $equiv['codigo_barras'];
                            $updateTypes .= "s";
                        }
                        if (isset($equiv['costo'])) {
                            $updateFields[] = "precio_compra = ?";
                            $updateValues[] = floatval($equiv['costo']);
                            $updateTypes .= "d";
                        }
                        if (isset($equiv['precio_lista'])) {
                            $updateFields[] = "precio_lista = ?";
                            $updateValues[] = floatval($equiv['precio_lista']);
                            $updateTypes .= "d";
                        }
                        for ($i = 1; $i <= 6; $i++) {
                            $precioKey = "precio$i";
                            if (isset($equiv[$precioKey])) {
                                $updateFields[] = "$precioKey = ?";
                                $updateValues[] = floatval($equiv[$precioKey]);
                                $updateTypes .= "d";
                            }
                        }

                        if (!empty($updateFields)) {
                            $updateValues[] = $producto_equiv_id;
                            $updateTypes .= "i";

                            $updateQuery = "UPDATE productos SET " . implode(", ", $updateFields) . " WHERE id = ?";
                            $updateStmt = $conn->prepare($updateQuery);
                            $updateStmt->bind_param($updateTypes, ...$updateValues);
                            $updateStmt->execute();
                        }
                    }
                }

                $conn->commit();
                sendResponse(true, "Equivalencias guardadas correctamente");
            } catch (Exception $e) {
                $conn->rollback();
                throw $e;
            }
        } catch (Exception $e) {
            error_log("Error al guardar equivalencias: " . $e->getMessage());
            sendResponse(false, "Error al guardar equivalencias: " . $e->getMessage());
        }
        break;

    case 'eliminarEquivalencia':
        try {
            $id = isset($datapost['id']) ? intval($datapost['id']) : 0;

            if ($id <= 0) {
                sendResponse(false, "ID de equivalencia requerido");
                break;
            }

            $query = "UPDATE producto_equivalencias SET activo = 0 WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $id);

            if ($stmt->execute()) {
                sendResponse(true, "Equivalencia eliminada correctamente");
            } else {
                sendResponse(false, "Error al eliminar equivalencia");
            }
        } catch (Exception $e) {
            error_log("Error al eliminar equivalencia: " . $e->getMessage());
            sendResponse(false, "Error al eliminar equivalencia");
        }
        break;

    case 'getEquivalenciasBidireccionales':
        try {
            $producto_id = isset($datapost['producto_id']) ? intval($datapost['producto_id']) : 0;

            if ($producto_id <= 0) {
                sendResponse(false, "ID de producto requerido");
                break;
            }

            // Consulta UNION para obtener equivalencias en ambas direcciones
            // Directa: donde producto_id = ? (el producto buscado es el base)
            // Inversa: donde producto_equivalente_id = ? (el producto buscado es equivalente de otro)
            $query = "
                (SELECT
                    pe.id,
                    pe.producto_equivalente_id as producto_relacionado_id,
                    p.nombre as producto_relacionado_nombre,
                    p.codigo_corto,
                    p.codigo_barras,
                    p.unidad_medida,
                    p.precio_lista,
                    p.precio1,
                    p.precio2,
                    p.precio3,
                    p.precio4,
                    p.precio5,
                    p.precio6,
                    pe.cantidad as factor,
                    'directa' as direccion
                FROM producto_equivalencias pe
                INNER JOIN productos p ON pe.producto_equivalente_id = p.id
                WHERE pe.producto_id = ? AND pe.activo = 1 AND p.activo = 1)

                UNION

                (SELECT
                    pe.id,
                    pe.producto_id as producto_relacionado_id,
                    p.nombre as producto_relacionado_nombre,
                    p.codigo_corto,
                    p.codigo_barras,
                    p.unidad_medida,
                    p.precio_lista,
                    p.precio1,
                    p.precio2,
                    p.precio3,
                    p.precio4,
                    p.precio5,
                    p.precio6,
                    1/pe.cantidad as factor,
                    'inversa' as direccion
                FROM producto_equivalencias pe
                INNER JOIN productos p ON pe.producto_id = p.id
                WHERE pe.producto_equivalente_id = ? AND pe.activo = 1 AND p.activo = 1)

                ORDER BY producto_relacionado_nombre ASC
            ";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("ii", $producto_id, $producto_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $equivalencias = [];
            while ($row = $result->fetch_assoc()) {
                $equivalencias[] = [
                    'id' => intval($row['id']),
                    'producto_relacionado_id' => intval($row['producto_relacionado_id']),
                    'producto_relacionado_nombre' => $row['producto_relacionado_nombre'],
                    'codigo_corto' => $row['codigo_corto'],
                    'codigo_barras' => $row['codigo_barras'],
                    'unidad_medida' => $row['unidad_medida'],
                    'precio_lista' => floatval($row['precio_lista']),
                    'precio1' => floatval($row['precio1']),
                    'precio2' => floatval($row['precio2']),
                    'precio3' => floatval($row['precio3']),
                    'precio4' => floatval($row['precio4']),
                    'precio5' => floatval($row['precio5']),
                    'precio6' => floatval($row['precio6']),
                    'factor' => floatval($row['factor']),
                    'direccion' => $row['direccion']
                ];
            }

            sendResponse(true, "Equivalencias bidireccionales obtenidas correctamente", $equivalencias);
        } catch (Exception $e) {
            error_log("Error al obtener equivalencias bidireccionales: " . $e->getMessage());
            sendResponse(false, "Error al obtener equivalencias bidireccionales");
        }
        break;

    default:
        sendResponse(false, "Operación no válida para equivalencias de productos");
        break;
}
?>
