<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'getReporteVentas':
        try {
            $fecha_inicio = isset($datapost['fecha_inicio']) ? trim($datapost['fecha_inicio']) : date('Y-m-d');
            $fecha_fin = isset($datapost['fecha_fin']) ? trim($datapost['fecha_fin']) : date('Y-m-d');
            $metodo_pago = isset($datapost['metodo_pago']) ? trim($datapost['metodo_pago']) : '';
            $estado = isset($datapost['estado']) ? trim($datapost['estado']) : '';

            // Construir consulta base
            $query = "SELECT
                v.id,
                v.folio,
                v.cliente_nombre,
                v.usuario_nombre,
                v.subtotal,
                v.impuesto,
                v.descuento,
                v.total,
                v.metodo_pago,
                v.estado,
                v.fecha_creacion,
                COUNT(vd.id) as total_productos
            FROM ventas v
            LEFT JOIN venta_detalles vd ON v.id = vd.venta_id
            WHERE DATE(v.fecha_creacion) BETWEEN ? AND ?";

            // Agregar filtros opcionales
            $params = [$fecha_inicio, $fecha_fin];
            $types = "ss";

            if (!empty($metodo_pago)) {
                $query .= " AND v.metodo_pago = ?";
                $params[] = $metodo_pago;
                $types .= "s";
            }

            if (!empty($estado)) {
                $query .= " AND v.estado = ?";
                $params[] = $estado;
                $types .= "s";
            }

            $query .= " GROUP BY v.id ORDER BY v.fecha_creacion DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param($types, ...$params);
            $stmt->execute();
            $result = $stmt->get_result();

            $ventas = [];
            $total_ventas = 0;
            $total_general = 0;

            while ($row = $result->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'cliente_nombre' => $row['cliente_nombre'] ?? 'Cliente General',
                    'usuario_nombre' => $row['usuario_nombre'],
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'estado' => $row['estado'],
                    'total_productos' => intval($row['total_productos']),
                    'fecha_creacion' => $row['fecha_creacion']
                ];

                if ($row['estado'] === 'completada') {
                    $total_general += floatval($row['total']);
                    $total_ventas++;
                }
            }

            $resumen = [
                'total_ventas' => $total_ventas,
                'total_general' => $total_general,
                'promedio' => $total_ventas > 0 ? $total_general / $total_ventas : 0,
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin
            ];

            sendResponse(true, "Reporte de ventas obtenido exitosamente", [
                'ventas' => $ventas,
                'resumen' => $resumen
            ]);

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener reporte de ventas: " . $e->getMessage());
        }
        break;

    case 'getReporteVentasPorMetodoPago':
        try {
            $fecha_inicio = isset($datapost['fecha_inicio']) ? trim($datapost['fecha_inicio']) : date('Y-m-d');
            $fecha_fin = isset($datapost['fecha_fin']) ? trim($datapost['fecha_fin']) : date('Y-m-d');

            $query = "SELECT
                v.metodo_pago,
                COUNT(v.id) as cantidad,
                SUM(v.total) as total
            FROM ventas v
            WHERE DATE(v.fecha_creacion) BETWEEN ? AND ?
            AND v.estado = 'completada'
            GROUP BY v.metodo_pago
            ORDER BY total DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("ss", $fecha_inicio, $fecha_fin);
            $stmt->execute();
            $result = $stmt->get_result();

            $metodos = [];
            while ($row = $result->fetch_assoc()) {
                $metodos[] = [
                    'metodo_pago' => $row['metodo_pago'],
                    'cantidad' => intval($row['cantidad']),
                    'total' => floatval($row['total'])
                ];
            }

            sendResponse(true, "Reporte por método de pago obtenido exitosamente", $metodos);

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener reporte por método de pago: " . $e->getMessage());
        }
        break;

    case 'getReporteProductosMasVendidos':
        try {
            $fecha_inicio = isset($datapost['fecha_inicio']) ? trim($datapost['fecha_inicio']) : date('Y-m-d');
            $fecha_fin = isset($datapost['fecha_fin']) ? trim($datapost['fecha_fin']) : date('Y-m-d');
            $limite = isset($datapost['limite']) ? intval($datapost['limite']) : 10;

            $query = "SELECT
                vd.producto_id,
                vd.producto_nombre,
                SUM(vd.cantidad) as cantidad_vendida,
                SUM(vd.total) as total_vendido,
                COUNT(DISTINCT vd.venta_id) as num_ventas
            FROM venta_detalles vd
            INNER JOIN ventas v ON vd.venta_id = v.id
            WHERE DATE(v.fecha_creacion) BETWEEN ? AND ?
            AND v.estado = 'completada'
            GROUP BY vd.producto_id, vd.producto_nombre
            ORDER BY cantidad_vendida DESC
            LIMIT ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("ssi", $fecha_inicio, $fecha_fin, $limite);
            $stmt->execute();
            $result = $stmt->get_result();

            $productos = [];
            while ($row = $result->fetch_assoc()) {
                $productos[] = [
                    'producto_id' => intval($row['producto_id']),
                    'producto_nombre' => $row['producto_nombre'],
                    'cantidad_vendida' => floatval($row['cantidad_vendida']),
                    'total_vendido' => floatval($row['total_vendido']),
                    'num_ventas' => intval($row['num_ventas'])
                ];
            }

            sendResponse(true, "Productos más vendidos obtenidos exitosamente", $productos);

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener productos más vendidos: " . $e->getMessage());
        }
        break;

    case 'getReporteVentasPorUsuario':
        try {
            $fecha_inicio = isset($datapost['fecha_inicio']) ? trim($datapost['fecha_inicio']) : date('Y-m-d');
            $fecha_fin = isset($datapost['fecha_fin']) ? trim($datapost['fecha_fin']) : date('Y-m-d');

            $query = "SELECT
                v.usuario_id,
                v.usuario_nombre,
                COUNT(v.id) as total_ventas,
                SUM(v.total) as total_vendido,
                AVG(v.total) as promedio_venta
            FROM ventas v
            WHERE DATE(v.fecha_creacion) BETWEEN ? AND ?
            AND v.estado = 'completada'
            GROUP BY v.usuario_id, v.usuario_nombre
            ORDER BY total_vendido DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("ss", $fecha_inicio, $fecha_fin);
            $stmt->execute();
            $result = $stmt->get_result();

            $usuarios = [];
            while ($row = $result->fetch_assoc()) {
                $usuarios[] = [
                    'usuario_id' => intval($row['usuario_id']),
                    'usuario_nombre' => $row['usuario_nombre'],
                    'total_ventas' => intval($row['total_ventas']),
                    'total_vendido' => floatval($row['total_vendido']),
                    'promedio_venta' => floatval($row['promedio_venta'])
                ];
            }

            sendResponse(true, "Reporte por usuario obtenido exitosamente", $usuarios);

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener reporte por usuario: " . $e->getMessage());
        }
        break;

    case 'getReporteResumenDiario':
        try {
            $fecha_inicio = isset($datapost['fecha_inicio']) ? trim($datapost['fecha_inicio']) : date('Y-m-d', strtotime('-30 days'));
            $fecha_fin = isset($datapost['fecha_fin']) ? trim($datapost['fecha_fin']) : date('Y-m-d');

            $query = "SELECT
                DATE(v.fecha_creacion) as fecha,
                COUNT(v.id) as total_ventas,
                SUM(CASE WHEN v.estado = 'completada' THEN v.total ELSE 0 END) as total_vendido,
                SUM(CASE WHEN v.estado = 'cancelada' THEN 1 ELSE 0 END) as ventas_canceladas
            FROM ventas v
            WHERE DATE(v.fecha_creacion) BETWEEN ? AND ?
            GROUP BY DATE(v.fecha_creacion)
            ORDER BY fecha DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("ss", $fecha_inicio, $fecha_fin);
            $stmt->execute();
            $result = $stmt->get_result();

            $resumen = [];
            while ($row = $result->fetch_assoc()) {
                $resumen[] = [
                    'fecha' => $row['fecha'],
                    'total_ventas' => intval($row['total_ventas']),
                    'total_vendido' => floatval($row['total_vendido']),
                    'ventas_canceladas' => intval($row['ventas_canceladas'])
                ];
            }

            sendResponse(true, "Resumen diario obtenido exitosamente", $resumen);

        } catch (Exception $e) {
            sendResponse(false, "Error al obtener resumen diario: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para reportes");
        break;
}
?>
