<?php
/**
 * RetirosEfectivo.php
 * Endpoints para gestionar retiros de efectivo de caja
 */

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

switch ($tipo) {
    case 'verificarPermisoRetiro':
        ret_verificarPermisoRetiro();
        break;
    case 'verificarPermisoDevolucion':
        ret_verificarPermisoDevolucion();
        break;
    case 'verificarPermisoPrecioLibre':
        ret_verificarPermisoPrecioLibre();
        break;
    case 'crearRetiroEfectivo':
        ret_crearRetiroEfectivo();
        break;
    case 'getRetirosEfectivo':
        ret_getRetirosEfectivo();
        break;
    case 'getRetirosPendientesCorte':
        ret_getRetirosPendientesCorte();
        break;
    default:
        sendResponse(false, "Operación de retiros no implementada: " . $tipo);
        break;
}

/**
 * Verificar contraseña y permiso para retiro de efectivo
 * Busca el usuario por contraseña y verifica si tiene el permiso
 */
function ret_verificarPermisoRetiro() {
    global $conn, $datapost;

    $password = isset($datapost['password']) ? $datapost['password'] : null;

    if (!$password) {
        sendResponse(false, "Contraseña requerida");
        return;
    }

    try {
        // Buscar todos los usuarios activos
        $stmtUsers = $conn->prepare("SELECT id, password, nombre, rol_id FROM usuarios WHERE activo = 1");
        $stmtUsers->execute();
        $usersResult = $stmtUsers->get_result();

        $usuarioEncontrado = null;

        // Buscar qué usuario tiene esta contraseña
        while ($user = $usersResult->fetch_assoc()) {
            if (password_verify($password, $user['password'])) {
                $usuarioEncontrado = $user;
                break;
            }
        }

        if (!$usuarioEncontrado) {
            sendResponse(false, "Contraseña incorrecta");
            return;
        }

        // Verificar permiso de retiro de efectivo para el usuario encontrado
        $stmtPermiso = $conn->prepare("
            SELECT 1 FROM rol_permisos rp
            INNER JOIN permisos p ON rp.permiso_id = p.id
            WHERE rp.rol_id = ? AND p.nombre = 'retiro_efectivo'
        ");
        $stmtPermiso->bind_param("i", $usuarioEncontrado['rol_id']);
        $stmtPermiso->execute();
        $permisoResult = $stmtPermiso->get_result();

        if ($permisoResult->num_rows === 0) {
            sendResponse(false, "El usuario " . $usuarioEncontrado['nombre'] . " no tiene permiso para realizar retiros de efectivo");
            return;
        }

        sendResponse(true, "Acceso autorizado", [
            'usuario_id' => $usuarioEncontrado['id'],
            'usuario_nombre' => $usuarioEncontrado['nombre']
        ]);

    } catch (Exception $e) {
        sendResponse(false, "Error al verificar permiso: " . $e->getMessage());
    }
}

/**
 * Verificar contraseña y permiso para devolución
 * Busca el usuario por contraseña y verifica si tiene el permiso devolucion_autorizar
 */
function ret_verificarPermisoDevolucion() {
    global $conn, $datapost;

    $password = isset($datapost['password']) ? $datapost['password'] : null;

    if (!$password) {
        sendResponse(false, "Contraseña requerida");
        return;
    }

    try {
        // Buscar todos los usuarios activos
        $stmtUsers = $conn->prepare("SELECT id, password, nombre, rol_id FROM usuarios WHERE activo = 1");
        $stmtUsers->execute();
        $usersResult = $stmtUsers->get_result();

        $usuarioEncontrado = null;

        // Buscar qué usuario tiene esta contraseña
        while ($user = $usersResult->fetch_assoc()) {
            if (password_verify($password, $user['password'])) {
                $usuarioEncontrado = $user;
                break;
            }
        }

        if (!$usuarioEncontrado) {
            sendResponse(false, "Contraseña incorrecta");
            return;
        }

        // Verificar permiso de devolución para el usuario encontrado
        $stmtPermiso = $conn->prepare("
            SELECT 1 FROM rol_permisos rp
            INNER JOIN permisos p ON rp.permiso_id = p.id
            WHERE rp.rol_id = ? AND p.nombre = 'devolucion_autorizar'
        ");
        $stmtPermiso->bind_param("i", $usuarioEncontrado['rol_id']);
        $stmtPermiso->execute();
        $permisoResult = $stmtPermiso->get_result();

        if ($permisoResult->num_rows === 0) {
            sendResponse(false, "El usuario " . $usuarioEncontrado['nombre'] . " no tiene permiso para autorizar devoluciones");
            return;
        }

        sendResponse(true, "Acceso autorizado", [
            'usuario_id' => $usuarioEncontrado['id'],
            'usuario_nombre' => $usuarioEncontrado['nombre']
        ]);

    } catch (Exception $e) {
        sendResponse(false, "Error al verificar permiso: " . $e->getMessage());
    }
}

/**
 * Verificar contraseña y permiso para precio libre
 * Busca el usuario por contraseña y verifica si tiene el permiso ventas_precio_libre
 */
function ret_verificarPermisoPrecioLibre() {
    global $conn, $datapost;

    $password = isset($datapost['password']) ? $datapost['password'] : null;

    if (!$password) {
        sendResponse(false, "Contraseña requerida");
        return;
    }

    try {
        // Buscar todos los usuarios activos
        $stmtUsers = $conn->prepare("SELECT id, password, nombre, rol_id FROM usuarios WHERE activo = 1");
        $stmtUsers->execute();
        $usersResult = $stmtUsers->get_result();

        $usuarioEncontrado = null;

        // Buscar qué usuario tiene esta contraseña
        while ($user = $usersResult->fetch_assoc()) {
            if (password_verify($password, $user['password'])) {
                $usuarioEncontrado = $user;
                break;
            }
        }

        if (!$usuarioEncontrado) {
            sendResponse(false, "Contraseña incorrecta");
            return;
        }

        // Verificar permiso de precio libre para el usuario encontrado
        $stmtPermiso = $conn->prepare("
            SELECT 1 FROM rol_permisos rp
            INNER JOIN permisos p ON rp.permiso_id = p.id
            WHERE rp.rol_id = ? AND p.nombre = 'ventas_precio_libre'
        ");
        $stmtPermiso->bind_param("i", $usuarioEncontrado['rol_id']);
        $stmtPermiso->execute();
        $permisoResult = $stmtPermiso->get_result();

        if ($permisoResult->num_rows === 0) {
            sendResponse(false, "El usuario " . $usuarioEncontrado['nombre'] . " no tiene permiso para capturar precio libre");
            return;
        }

        sendResponse(true, "Acceso autorizado", [
            'usuario_id' => $usuarioEncontrado['id'],
            'usuario_nombre' => $usuarioEncontrado['nombre']
        ]);

    } catch (Exception $e) {
        sendResponse(false, "Error al verificar permiso: " . $e->getMessage());
    }
}

/**
 * Crear un retiro de efectivo
 */
function ret_crearRetiroEfectivo() {
    global $conn, $datapost;

    $usuarioId = isset($datapost['usuario_id']) ? (int)$datapost['usuario_id'] : null;
    $monto = isset($datapost['monto']) ? (float)$datapost['monto'] : 0;
    $motivo = isset($datapost['motivo']) ? trim($datapost['motivo']) : null;
    $nombrePc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : null;

    // Validaciones
    if (!$usuarioId) {
        sendResponse(false, "Usuario requerido");
        return;
    }

    if ($monto <= 0) {
        sendResponse(false, "El monto debe ser mayor a 0");
        return;
    }

    if (!$nombrePc) {
        sendResponse(false, "Nombre de PC requerido para identificar la caja");
        return;
    }

    try {
        // Obtener caja_id por nombre_pc
        $stmtCaja = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
        $stmtCaja->bind_param("s", $nombrePc);
        $stmtCaja->execute();
        $cajaResult = $stmtCaja->get_result();

        if ($cajaResult->num_rows === 0) {
            sendResponse(false, "Caja no encontrada para este equipo");
            return;
        }

        $cajaId = $cajaResult->fetch_assoc()['id'];

        // Insertar retiro
        $stmt = $conn->prepare("
            INSERT INTO retiros_efectivo (caja_id, usuario_id, monto, motivo, nombre_pc, fecha_retiro, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, NOW(), NOW(), NOW())
        ");
        $stmt->bind_param("iidss", $cajaId, $usuarioId, $monto, $motivo, $nombrePc);

        if ($stmt->execute()) {
            $retiroId = $conn->insert_id;

            sendResponse(true, "Retiro registrado correctamente", [
                'id' => $retiroId,
                'monto' => $monto,
                'motivo' => $motivo,
                'fecha' => date('Y-m-d H:i:s')
            ]);
        } else {
            throw new Exception("Error al registrar retiro: " . $conn->error);
        }
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener retiros de efectivo (con filtros opcionales)
 */
function ret_getRetirosEfectivo() {
    global $conn, $datapost;

    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);
    $fechaInicio = isset($_GET['fecha_inicio']) ? $_GET['fecha_inicio'] : (isset($datapost['fecha_inicio']) ? $datapost['fecha_inicio'] : null);
    $fechaFin = isset($_GET['fecha_fin']) ? $_GET['fecha_fin'] : (isset($datapost['fecha_fin']) ? $datapost['fecha_fin'] : null);

    try {
        $query = "
            SELECT
                r.*,
                u.nombre as usuario_nombre,
                c.identificador as caja_nombre
            FROM retiros_efectivo r
            LEFT JOIN usuarios u ON r.usuario_id = u.id
            LEFT JOIN cajas c ON r.caja_id = c.id
            WHERE 1=1
        ";
        $params = [];
        $types = "";

        if ($nombrePc) {
            $query .= " AND r.nombre_pc = ?";
            $params[] = $nombrePc;
            $types .= "s";
        }

        if ($fechaInicio) {
            $query .= " AND DATE(r.fecha_retiro) >= ?";
            $params[] = $fechaInicio;
            $types .= "s";
        }

        if ($fechaFin) {
            $query .= " AND DATE(r.fecha_retiro) <= ?";
            $params[] = $fechaFin;
            $types .= "s";
        }

        $query .= " ORDER BY r.fecha_retiro DESC LIMIT 100";

        $stmt = $conn->prepare($query);
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        $result = $stmt->get_result();

        $retiros = [];
        while ($row = $result->fetch_assoc()) {
            $retiros[] = $row;
        }

        sendResponse(true, "Retiros obtenidos", $retiros);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}

/**
 * Obtener total de retiros desde el último corte de caja
 */
function ret_getRetirosPendientesCorte() {
    global $conn, $datapost;

    $nombrePc = isset($_GET['nombre_pc']) ? $_GET['nombre_pc'] : (isset($datapost['nombre_pc']) ? $datapost['nombre_pc'] : null);

    if (!$nombrePc) {
        sendResponse(false, "Nombre de PC requerido");
        return;
    }

    try {
        // Obtener caja_id
        $stmtCaja = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
        $stmtCaja->bind_param("s", $nombrePc);
        $stmtCaja->execute();
        $cajaResult = $stmtCaja->get_result();

        if ($cajaResult->num_rows === 0) {
            sendResponse(false, "Caja no encontrada");
            return;
        }

        $cajaId = $cajaResult->fetch_assoc()['id'];

        // Obtener fecha del último corte
        $stmtCorte = $conn->prepare("
            SELECT fecha_creacion FROM cortes_caja
            WHERE caja_id = ?
            ORDER BY fecha_creacion DESC
            LIMIT 1
        ");
        $stmtCorte->bind_param("i", $cajaId);
        $stmtCorte->execute();
        $corteResult = $stmtCorte->get_result();

        $fechaUltimoCorte = null;
        if ($corteResult->num_rows > 0) {
            $fechaUltimoCorte = $corteResult->fetch_assoc()['fecha_creacion'];
        }

        // Obtener total de retiros desde el último corte
        if ($fechaUltimoCorte) {
            $stmtRetiros = $conn->prepare("
                SELECT
                    COALESCE(SUM(monto), 0) as total_retiros,
                    COUNT(*) as cantidad_retiros
                FROM retiros_efectivo
                WHERE caja_id = ? AND fecha_retiro > ?
            ");
            $stmtRetiros->bind_param("is", $cajaId, $fechaUltimoCorte);
        } else {
            $stmtRetiros = $conn->prepare("
                SELECT
                    COALESCE(SUM(monto), 0) as total_retiros,
                    COUNT(*) as cantidad_retiros
                FROM retiros_efectivo
                WHERE caja_id = ?
            ");
            $stmtRetiros->bind_param("i", $cajaId);
        }

        $stmtRetiros->execute();
        $retirosResult = $stmtRetiros->get_result();
        $totales = $retirosResult->fetch_assoc();

        sendResponse(true, "Retiros pendientes de corte", [
            'total_retiros' => (float)$totales['total_retiros'],
            'cantidad_retiros' => (int)$totales['cantidad_retiros'],
            'ultimo_corte' => $fechaUltimoCorte
        ]);
    } catch (Exception $e) {
        sendResponse(false, $e->getMessage());
    }
}
