<?php
// Verificar que se accede desde el archivo principal
if (!defined('BASEPATH')) {
    die('Acceso directo no permitido');
}

// Obtener rutas por rango de fechas
if ($tipo === 'getRutasEntrega') {
    $fecha_inicio = isset($datapost['fecha_inicio']) ? $datapost['fecha_inicio'] : date('Y-m-d');
    $fecha_fin = isset($datapost['fecha_fin']) ? $datapost['fecha_fin'] : date('Y-m-d');

    $sql = "SELECT r.*,
            (SELECT COUNT(*) FROM ruta_pedidos WHERE ruta_id = r.id) as total_pedidos
            FROM rutas_entrega r
            WHERE r.fecha BETWEEN ? AND ?
            ORDER BY r.fecha ASC, r.created_at ASC";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $fecha_inicio, $fecha_fin);
    $stmt->execute();
    $result = $stmt->get_result();

    $rutas = [];
    while ($row = $result->fetch_assoc()) {
        $rutas[] = $row;
    }

    sendResponse(true, 'Rutas obtenidas', $rutas);
}

// Obtener una ruta con sus pedidos
if ($tipo === 'getRutaEntrega') {
    $ruta_id = isset($datapost['id']) ? intval($datapost['id']) : 0;

    if ($ruta_id <= 0) {
        sendResponse(false, 'ID de ruta no válido');
    }

    // Obtener datos de la ruta
    $sql = "SELECT * FROM rutas_entrega WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $ruta_id);
    $stmt->execute();
    $ruta = $stmt->get_result()->fetch_assoc();

    if (!$ruta) {
        sendResponse(false, 'Ruta no encontrada');
    }

    // Obtener pedidos de la ruta con coordenadas
    $sql = "SELECT rp.id as ruta_pedido_id, rp.orden, rp.venta_id,
            v.folio, v.cliente_nombre, v.direccion_entrega, v.total,
            v.metodo_pago, v.estatus_pago, v.tipo_entrega, v.estado,
            v.fecha_creacion, v.cliente_id,
            cd.latitud, cd.longitud
            FROM ruta_pedidos rp
            INNER JOIN ventas v ON rp.venta_id = v.id
            LEFT JOIN cliente_direcciones cd ON v.cliente_id = cd.cliente_id
                AND cd.direccion = v.direccion_entrega
                AND cd.activo = 1
            WHERE rp.ruta_id = ?
            ORDER BY rp.orden ASC";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $ruta_id);
    $stmt->execute();
    $result = $stmt->get_result();

    $pedidos = [];
    while ($row = $result->fetch_assoc()) {
        // Convertir coordenadas a float si existen
        if ($row['latitud']) {
            $row['latitud'] = floatval($row['latitud']);
        }
        if ($row['longitud']) {
            $row['longitud'] = floatval($row['longitud']);
        }
        $pedidos[] = $row;
    }

    $ruta['pedidos'] = $pedidos;

    sendResponse(true, 'Ruta obtenida', $ruta);
}

// Crear nueva ruta
if ($tipo === 'crearRutaEntrega') {
    $nombre = isset($datapost['nombre']) ? trim($datapost['nombre']) : '';
    $fecha = isset($datapost['fecha']) ? $datapost['fecha'] : date('Y-m-d');

    if (empty($nombre)) {
        sendResponse(false, 'El nombre de la ruta es requerido');
    }

    $sql = "INSERT INTO rutas_entrega (nombre, fecha, estado) VALUES (?, ?, 'pendiente')";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $nombre, $fecha);

    if ($stmt->execute()) {
        $ruta_id = $conn->insert_id;
        sendResponse(true, 'Ruta creada exitosamente', ['id' => $ruta_id, 'nombre' => $nombre, 'fecha' => $fecha]);
    } else {
        sendResponse(false, 'Error al crear la ruta: ' . $conn->error);
    }
}

// Actualizar ruta
if ($tipo === 'actualizarRutaEntrega') {
    $id = isset($datapost['id']) ? intval($datapost['id']) : 0;
    $nombre = isset($datapost['nombre']) ? trim($datapost['nombre']) : '';
    $estado = isset($datapost['estado']) ? $datapost['estado'] : null;

    if ($id <= 0) {
        sendResponse(false, 'ID de ruta no válido');
    }

    $updates = [];
    $params = [];
    $types = "";

    if (!empty($nombre)) {
        $updates[] = "nombre = ?";
        $params[] = $nombre;
        $types .= "s";
    }

    if ($estado !== null) {
        $updates[] = "estado = ?";
        $params[] = $estado;
        $types .= "s";
    }

    if (empty($updates)) {
        sendResponse(false, 'No hay datos para actualizar');
    }

    $params[] = $id;
    $types .= "i";

    $sql = "UPDATE rutas_entrega SET " . implode(", ", $updates) . " WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);

    if ($stmt->execute()) {
        sendResponse(true, 'Ruta actualizada exitosamente');
    } else {
        sendResponse(false, 'Error al actualizar la ruta: ' . $conn->error);
    }
}

// Eliminar ruta
if ($tipo === 'eliminarRutaEntrega') {
    $id = isset($datapost['id']) ? intval($datapost['id']) : 0;

    if ($id <= 0) {
        sendResponse(false, 'ID de ruta no válido');
    }

    $sql = "DELETE FROM rutas_entrega WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);

    if ($stmt->execute()) {
        sendResponse(true, 'Ruta eliminada exitosamente');
    } else {
        sendResponse(false, 'Error al eliminar la ruta: ' . $conn->error);
    }
}

// Agregar pedidos a una ruta
if ($tipo === 'agregarPedidosARuta') {
    $ruta_id = isset($datapost['ruta_id']) ? intval($datapost['ruta_id']) : 0;
    $venta_ids = isset($datapost['venta_ids']) ? $datapost['venta_ids'] : [];

    if ($ruta_id <= 0) {
        sendResponse(false, 'ID de ruta no válido');
    }

    if (empty($venta_ids) || !is_array($venta_ids)) {
        sendResponse(false, 'Debe seleccionar al menos un pedido');
    }

    // Obtener el orden máximo actual
    $sql = "SELECT COALESCE(MAX(orden), 0) as max_orden FROM ruta_pedidos WHERE ruta_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $ruta_id);
    $stmt->execute();
    $max_orden = $stmt->get_result()->fetch_assoc()['max_orden'];

    $agregados = 0;
    $errores = [];

    foreach ($venta_ids as $venta_id) {
        $venta_id = intval($venta_id);
        $max_orden++;

        // Verificar que el pedido no esté ya en otra ruta
        $sql = "SELECT id FROM ruta_pedidos WHERE venta_id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $venta_id);
        $stmt->execute();

        if ($stmt->get_result()->num_rows > 0) {
            $errores[] = "Pedido $venta_id ya está asignado a una ruta";
            continue;
        }

        $sql = "INSERT INTO ruta_pedidos (ruta_id, venta_id, orden) VALUES (?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("iii", $ruta_id, $venta_id, $max_orden);

        if ($stmt->execute()) {
            $agregados++;
        } else {
            $errores[] = "Error al agregar pedido $venta_id";
        }
    }

    if ($agregados > 0) {
        $mensaje = "$agregados pedido(s) agregado(s) a la ruta";
        if (!empty($errores)) {
            $mensaje .= ". Errores: " . implode(", ", $errores);
        }
        sendResponse(true, $mensaje);
    } else {
        sendResponse(false, 'No se pudo agregar ningún pedido. ' . implode(", ", $errores));
    }
}

// Actualizar orden de pedidos en la ruta (para drag & drop)
if ($tipo === 'actualizarOrdenRuta') {
    $ruta_id = isset($datapost['ruta_id']) ? intval($datapost['ruta_id']) : 0;
    $pedidos = isset($datapost['pedidos']) ? $datapost['pedidos'] : [];

    if ($ruta_id <= 0) {
        sendResponse(false, 'ID de ruta no válido');
    }

    if (empty($pedidos) || !is_array($pedidos)) {
        sendResponse(false, 'Lista de pedidos no válida');
    }

    $conn->begin_transaction();

    try {
        foreach ($pedidos as $index => $pedido) {
            $venta_id = intval($pedido['venta_id']);
            $orden = $index + 1;

            $sql = "UPDATE ruta_pedidos SET orden = ? WHERE ruta_id = ? AND venta_id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("iii", $orden, $ruta_id, $venta_id);
            $stmt->execute();
        }

        $conn->commit();
        sendResponse(true, 'Orden actualizado exitosamente');
    } catch (Exception $e) {
        $conn->rollback();
        sendResponse(false, 'Error al actualizar el orden: ' . $e->getMessage());
    }
}

// Quitar pedido de una ruta
if ($tipo === 'quitarPedidoDeRuta') {
    $ruta_id = isset($datapost['ruta_id']) ? intval($datapost['ruta_id']) : 0;
    $venta_id = isset($datapost['venta_id']) ? intval($datapost['venta_id']) : 0;

    if ($ruta_id <= 0 || $venta_id <= 0) {
        sendResponse(false, 'IDs no válidos');
    }

    $sql = "DELETE FROM ruta_pedidos WHERE ruta_id = ? AND venta_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $ruta_id, $venta_id);

    if ($stmt->execute()) {
        // Reordenar los pedidos restantes
        $sql = "SET @orden = 0;
                UPDATE ruta_pedidos SET orden = (@orden := @orden + 1)
                WHERE ruta_id = ? ORDER BY orden";
        $conn->query("SET @orden = 0");
        $stmt = $conn->prepare("UPDATE ruta_pedidos SET orden = (@orden := @orden + 1) WHERE ruta_id = ? ORDER BY orden");
        $stmt->bind_param("i", $ruta_id);
        $stmt->execute();

        sendResponse(true, 'Pedido quitado de la ruta');
    } else {
        sendResponse(false, 'Error al quitar el pedido: ' . $conn->error);
    }
}

// Obtener pedidos a domicilio sin ruta asignada
if ($tipo === 'getPedidosDomicilioSinRuta') {
    $fecha_inicio = isset($datapost['fecha_inicio']) ? $datapost['fecha_inicio'] : date('Y-m-d');
    $fecha_fin = isset($datapost['fecha_fin']) ? $datapost['fecha_fin'] : date('Y-m-d');

    $sql = "SELECT v.id, v.folio, v.cliente_nombre, v.direccion_entrega,
            v.total, v.metodo_pago, v.estatus_pago, v.estado, v.fecha_creacion,
            v.tipo_entrega
            FROM ventas v
            LEFT JOIN ruta_pedidos rp ON v.id = rp.venta_id
            WHERE v.tipo_entrega = 'domicilio'
            AND DATE(v.fecha_creacion) BETWEEN ? AND ?
            AND rp.id IS NULL
            ORDER BY v.fecha_creacion ASC";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $fecha_inicio, $fecha_fin);
    $stmt->execute();
    $result = $stmt->get_result();

    $pedidos = [];
    while ($row = $result->fetch_assoc()) {
        $pedidos[] = $row;
    }

    sendResponse(true, 'Pedidos sin ruta obtenidos', $pedidos);
}

// Generar carátula para imprimir
if ($tipo === 'imprimirCaratula') {
    $ruta_id = isset($datapost['ruta_id']) ? intval($datapost['ruta_id']) : 0;

    if ($ruta_id <= 0) {
        sendResponse(false, 'ID de ruta no válido');
    }

    // Obtener datos de la ruta
    $sql = "SELECT * FROM rutas_entrega WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $ruta_id);
    $stmt->execute();
    $ruta = $stmt->get_result()->fetch_assoc();

    if (!$ruta) {
        sendResponse(false, 'Ruta no encontrada');
    }

    // Obtener logo de la empresa (configuración global)
    $logo_base64 = '';
    $sql_logo = "SELECT valor FROM configuraciones WHERE clave = 'empresa_logo' AND caja_id IS NULL";
    $result_logo = $conn->query($sql_logo);
    if ($result_logo && $row_logo = $result_logo->fetch_assoc()) {
        $logo_base64 = $row_logo['valor'];
    }

    // Obtener pedidos de la ruta con información completa (incluyendo teléfonos)
    $sql = "SELECT rp.orden, rp.venta_id, v.folio, v.cliente_nombre, v.direccion_entrega,
            v.total, v.metodo_pago, v.estatus_pago,
            (SELECT GROUP_CONCAT(ct.telefono SEPARATOR ', ')
             FROM cliente_telefonos ct
             WHERE ct.cliente_id = v.cliente_id AND ct.activo = 1
             ORDER BY ct.tipo = 'principal' DESC) as cliente_telefonos
            FROM ruta_pedidos rp
            INNER JOIN ventas v ON rp.venta_id = v.id
            WHERE rp.ruta_id = ?
            ORDER BY rp.orden ASC";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $ruta_id);
    $stmt->execute();
    $result = $stmt->get_result();

    $pedidos = [];
    while ($row = $result->fetch_assoc()) {
        $pedidos[] = $row;
    }

    // Formatear fecha
    $fecha_formateada = date('d/m/Y', strtotime($ruta['fecha']));

    // Función para traducir método de pago
    $traducirMetodoPago = function($metodo) {
        $metodos = [
            'efectivo' => 'Efectivo',
            'tarjeta' => 'Tarjeta',
            'transferencia' => 'Transferencia',
            'credito' => 'Crédito',
            'mixto' => 'Mixto'
        ];
        return $metodos[$metodo] ?? ucfirst($metodo ?? 'N/A');
    };

    // Generar HTML del logo si existe
    $logoHtml = '';
    if (!empty($logo_base64)) {
        $logoHtml = "<img src='{$logo_base64}' style='max-height: 70px; max-width: 150px;' alt='Logo' />";
    }

    // Generar HTML para impresión (formato horizontal con códigos de barras)
    $html = "
    <style>
        @page { size: landscape; margin: 10mm; }
        @media print {
            body { margin: 0; }
            .no-print { display: none; }
        }
        .barcode-svg { height: 35px; }
    </style>
    <script src='https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js'></script>

    <div style='font-family: Arial, sans-serif; margin: 0 auto; padding: 15px;'>
        <div style='display: flex; justify-content: space-between; align-items: center; border-bottom: 2px solid #333; padding-bottom: 10px; margin-bottom: 15px;'>
            <div style='flex: 1;'></div>
            <div style='text-align: center; flex: 2;'>
                <h1 style='margin: 0; font-size: 22px;'>RUTA DE ENTREGA</h1>
                <h2 style='margin: 5px 0; font-size: 16px; color: #666;'>{$ruta['nombre']}</h2>
                <p style='margin: 5px 0; font-size: 13px;'>Fecha: {$fecha_formateada}</p>
            </div>
            <div style='flex: 1; text-align: right;'>
                {$logoHtml}
            </div>
        </div>

        <table style='width: 100%; border-collapse: collapse; font-size: 11px;'>
            <thead>
                <tr style='background: #f0f0f0;'>
                    <th style='border: 1px solid #333; padding: 5px; width: 25px;'>#</th>
                    <th style='border: 1px solid #333; padding: 5px; width: 120px;'>CÓDIGO</th>
                    <th style='border: 1px solid #333; padding: 5px; width: 55px;'>FOLIO</th>
                    <th style='border: 1px solid #333; padding: 5px;'>CLIENTE</th>
                    <th style='border: 1px solid #333; padding: 5px;'>DIRECCIÓN</th>
                    <th style='border: 1px solid #333; padding: 5px; width: 100px;'>TELÉFONO</th>
                    <th style='border: 1px solid #333; padding: 5px; width: 75px;'>PAGO</th>
                    <th style='border: 1px solid #333; padding: 5px; width: 65px;'>MONTO</th>
                </tr>
            </thead>
            <tbody>";

    $totalMonto = 0;
    foreach ($pedidos as $pedido) {
        $metodoPago = $traducirMetodoPago($pedido['metodo_pago']);
        $estatusPago = $pedido['estatus_pago'] === 'pagado' ? '✓' : '(Pend.)';
        $pagoInfo = $metodoPago . ' ' . $estatusPago;
        $telefono = $pedido['cliente_telefonos'] ?? '-';
        $monto = number_format($pedido['total'], 2);
        $totalMonto += floatval($pedido['total']);
        $ventaId = $pedido['venta_id'];

        $html .= "
                <tr>
                    <td style='border: 1px solid #333; padding: 4px; text-align: center; font-weight: bold;'>{$pedido['orden']}</td>
                    <td style='border: 1px solid #333; padding: 4px; text-align: center;'>
                        <svg class='barcode-svg' id='barcode-{$ventaId}'></svg>
                    </td>
                    <td style='border: 1px solid #333; padding: 4px; text-align: center;'>#{$pedido['folio']}</td>
                    <td style='border: 1px solid #333; padding: 4px;'>{$pedido['cliente_nombre']}</td>
                    <td style='border: 1px solid #333; padding: 4px; font-size: 10px;'>{$pedido['direccion_entrega']}</td>
                    <td style='border: 1px solid #333; padding: 4px; text-align: center;'>{$telefono}</td>
                    <td style='border: 1px solid #333; padding: 4px; text-align: center;'>{$pagoInfo}</td>
                    <td style='border: 1px solid #333; padding: 4px; text-align: right;'>\${$monto}</td>
                </tr>";
    }

    $totalMontoFormateado = number_format($totalMonto, 2);

    // Generar array de venta_ids para el script
    $ventaIds = array_map(function($p) { return $p['venta_id']; }, $pedidos);
    $ventaIdsJson = json_encode($ventaIds);

    $html .= "
            </tbody>
        </table>

        <div style='margin-top: 15px; display: flex; justify-content: space-between; font-size: 12px; font-weight: bold; padding: 8px; background: #f5f5f5; border-radius: 4px;'>
            <span>Total de entregas: " . count($pedidos) . "</span>
            <span>Monto total: \${$totalMontoFormateado}</span>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            var ventaIds = {$ventaIdsJson};
            ventaIds.forEach(function(id) {
                try {
                    JsBarcode('#barcode-' + id, 'V' + id, {
                        format: 'CODE128',
                        width: 1.5,
                        height: 30,
                        displayValue: true,
                        fontSize: 10,
                        margin: 2
                    });
                } catch(e) {
                    console.error('Error generando código de barras para V' + id, e);
                }
            });
        });
    </script>";

    sendResponse(true, 'Carátula generada', [
        'ruta' => $ruta,
        'pedidos' => $pedidos,
        'html' => $html
    ]);
}
