<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'imprimirTicket':
    case 'generarTicket':
        $venta_id = intval($datapost['venta_id'] ?? 0);

        if ($venta_id <= 0) {
            sendResponse(false, "ID de venta requerido");
        }

        try {
            // Obtener datos de la venta
            $query = "SELECT
                v.id,
                v.folio,
                v.cliente_id,
                v.cliente_nombre,
                v.usuario_id,
                v.usuario_nombre,
                v.subtotal,
                v.impuesto,
                v.descuento,
                v.total,
                v.metodo_pago,
                v.tipo_entrega,
                v.direccion_entrega,
                v.estado,
                v.observaciones,
                v.fecha_creacion
            FROM ventas v
            WHERE v.id = ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $venta_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                // Obtener detalles de la venta
                $detallesQuery = "SELECT
                    id,
                    producto_id,
                    producto_nombre,
                    cantidad,
                    precio_unitario,
                    subtotal,
                    impuesto,
                    descuento,
                    total
                FROM venta_detalles
                WHERE venta_id = ?
                ORDER BY id ASC";

                $detallesStmt = $conn->prepare($detallesQuery);
                $detallesStmt->bind_param("i", $venta_id);
                $detallesStmt->execute();
                $detallesResult = $detallesStmt->get_result();

                $detalles = [];
                while ($detRow = $detallesResult->fetch_assoc()) {
                    $detalles[] = [
                        'id' => intval($detRow['id']),
                        'producto_id' => intval($detRow['producto_id']),
                        'producto_nombre' => $detRow['producto_nombre'],
                        'cantidad' => floatval($detRow['cantidad']),
                        'precio_unitario' => floatval($detRow['precio_unitario']),
                        'subtotal' => floatval($detRow['subtotal']),
                        'impuesto' => floatval($detRow['impuesto']),
                        'descuento' => floatval($detRow['descuento']),
                        'total' => floatval($detRow['total'])
                    ];
                }

                // Generar ticket en formato texto
                $ticket = generarTicketTexto([
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'cliente_id' => $row['cliente_id'] ? intval($row['cliente_id']) : null,
                    'cliente_nombre' => $row['cliente_nombre'] ?? 'Cliente General',
                    'usuario_id' => intval($row['usuario_id']),
                    'usuario_nombre' => $row['usuario_nombre'],
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'tipo_entrega' => $row['tipo_entrega'],
                    'direccion_entrega' => $row['direccion_entrega'],
                    'estado' => $row['estado'],
                    'observaciones' => $row['observaciones'],
                    'detalles' => $detalles,
                    'fecha_creacion' => $row['fecha_creacion']
                ]);

                sendResponse(true, "Ticket generado exitosamente", [
                    'ticket' => $ticket,
                    'venta_id' => $venta_id,
                    'folio' => $row['folio']
                ]);
            } else {
                sendResponse(false, "Venta no encontrada");
            }

        } catch (Exception $e) {
            sendResponse(false, "Error al generar ticket: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para tickets");
        break;
}

// Función para generar ticket en formato texto
function generarTicketTexto($venta) {
    $ancho = 35; // Ancho del ticket en caracteres (ajustado para impresora térmica)
    $linea = str_repeat('=', $ancho);
    $lineaPunteada = str_repeat('-', $ancho);

    // Encabezado
    $ticket = "\n";
    $ticket .= centrarTexto("ABARROTES", $ancho) . "\n";
    $ticket .= centrarTexto("PUNTO DE VENTA", $ancho) . "\n";
    $ticket .= $linea . "\n";

    // Información de la venta
    $ticket .= "Folio: " . $venta['folio'] . "\n";
    $ticket .= "Fecha: " . date('d/m/Y H:i:s', strtotime($venta['fecha_creacion'])) . "\n";
    $ticket .= "Cliente: " . $venta['cliente_nombre'] . "\n";
    $ticket .= "Atendido por: " . $venta['usuario_nombre'] . "\n";

    // Información de entrega
    if ($venta['tipo_entrega'] === 'domicilio') {
        $ticket .= "ENTREGA: DOMICILIO\n";
        if (!empty($venta['direccion_entrega'])) {
            $ticket .= "Direccion:\n";
            $ticket .= wordwrap($venta['direccion_entrega'], $ancho - 2, "\n", true) . "\n";
        }
    } else {
        $ticket .= "ENTREGA: PICKUP (Recoger en tienda)\n";
    }

    $ticket .= $linea . "\n";

    // Encabezado de productos
    $ticket .= ajustarColumnas("CANT", 5) . ajustarColumnas("PRODUCTO", 10) . ajustarColumnas("PRECIO", 10, 'right') . ajustarColumnas("TOTAL", 10, 'right') . "\n";
    $ticket .= $lineaPunteada . "\n";

    // Detalles de productos
    foreach ($venta['detalles'] as $detalle) {
        $nombreProducto = $detalle['producto_nombre'];
        $maxCaracteresLinea1 = 10;

        // Primera línea con cantidad, inicio del nombre, precio y total
        $nombre1 = substr($nombreProducto, 0, $maxCaracteresLinea1);
        $ticket .= ajustarColumnas(number_format($detalle['cantidad'], 2), 5) .
                   ajustarColumnas($nombre1, 10) .
                   ajustarColumnas('$' . number_format($detalle['precio_unitario'], 2), 10, 'right') .
                   ajustarColumnas('$' . number_format($detalle['total'], 2), 10, 'right') . "\n";

        // Líneas adicionales con el resto del nombre si es necesario (máximo 10 caracteres por línea)
        if (strlen($nombreProducto) > $maxCaracteresLinea1) {
            $nombre2 = substr($nombreProducto, $maxCaracteresLinea1);
            // Dividir en líneas de 10 caracteres
            $lineasNombre = str_split($nombre2, 10);
            foreach ($lineasNombre as $lineaNombre) {
                $ticket .= str_repeat(' ', 5) . // Espacios para alinear con la columna del producto
                           ajustarColumnas($lineaNombre, 10) . "\n";
            }
        }
    }

    $ticket .= $lineaPunteada . "\n";

    // Totales
    $ticket .= ajustarColumnas("SUBTOTAL:", 25, 'right') . ajustarColumnas('$' . number_format($venta['subtotal'], 2), 10, 'right') . "\n";

    if ($venta['descuento'] > 0) {
        $ticket .= ajustarColumnas("DESCUENTO:", 25, 'right') . ajustarColumnas('-$' . number_format($venta['descuento'], 2), 10, 'right') . "\n";
    }

    if ($venta['impuesto'] > 0) {
        $ticket .= ajustarColumnas("IMPUESTO:", 25, 'right') . ajustarColumnas('$' . number_format($venta['impuesto'], 2), 10, 'right') . "\n";
    }

    $ticket .= $linea . "\n";
    $ticket .= ajustarColumnas("TOTAL:", 25, 'right', true) . ajustarColumnas('$' . number_format($venta['total'], 2), 10, 'right', true) . "\n";
    $ticket .= $linea . "\n";

    // Observaciones
    if (!empty($venta['observaciones'])) {
        $ticket .= $lineaPunteada . "\n";
        $ticket .= "Observaciones:\n" . wordwrap($venta['observaciones'], $ancho - 2) . "\n";
    }

    // Pie de página
    $ticket .= $linea . "\n";
    $ticket .= centrarTexto("GRACIAS POR SU COMPRA", $ancho) . "\n";
    $ticket .= centrarTexto("¡VUELVA PRONTO!", $ancho) . "\n";
    $ticket .= $linea . "\n";
    $ticket .= "\n\n\n"; // Espacio para corte

    return $ticket;
}

// Función auxiliar para centrar texto
function centrarTexto($texto, $ancho) {
    $len = strlen($texto);
    if ($len >= $ancho) return substr($texto, 0, $ancho);

    $espacios = floor(($ancho - $len) / 2);
    return str_repeat(' ', $espacios) . $texto;
}

// Función auxiliar para ajustar columnas
function ajustarColumnas($texto, $ancho, $alineacion = 'left', $bold = false) {
    $len = strlen($texto);

    if ($len >= $ancho) {
        return substr($texto, 0, $ancho);
    }

    if ($alineacion === 'right') {
        return str_repeat(' ', $ancho - $len) . $texto;
    } else if ($alineacion === 'center') {
        $espacios = floor(($ancho - $len) / 2);
        return str_repeat(' ', $espacios) . $texto . str_repeat(' ', $ancho - $len - $espacios);
    } else {
        return $texto . str_repeat(' ', $ancho - $len);
    }
}

?>
