<?php
// Prevenir acceso directo
if (!defined('BASEPATH')) exit('No direct script access allowed');

switch ($tipo) {
    case 'getVentas':
        try {
            $query = "SELECT
                v.id,
                v.folio,
                v.cliente_id,
                v.cliente_nombre,
                c.nombre_comercial as cliente_nombre_comercial,
                c.tiene_credito,
                c.monto_credito_maximo,
                c.saldo_credito_usado,
                v.usuario_id,
                v.usuario_nombre,
                v.subtotal,
                v.impuesto,
                v.descuento,
                v.total,
                v.metodo_pago,
                v.estatus_pago,
                v.tipo_entrega,
                v.direccion_entrega,
                v.estado,
                v.observaciones,
                v.fecha_creacion,
                v.fecha_actualizacion,
                COUNT(vd.id) as total_productos
            FROM ventas v
            LEFT JOIN venta_detalles vd ON v.id = vd.venta_id
            LEFT JOIN clientes c ON v.cliente_id = c.id
            GROUP BY v.id
            ORDER BY v.fecha_creacion DESC";

            $result = $conn->query($query);

            if (!$result) {
                sendResponse(false, "Error en consulta: " . $conn->error);
                break;
            }

            $ventas = [];
            while ($row = $result->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'cliente_id' => $row['cliente_id'] ? intval($row['cliente_id']) : null,
                    'cliente_nombre' => $row['cliente_nombre'] ?? 'Cliente General',
                    'cliente_nombre_comercial' => $row['cliente_nombre_comercial'],
                    'cliente_tiene_credito' => $row['tiene_credito'] ? intval($row['tiene_credito']) : 0,
                    'cliente_monto_credito_maximo' => $row['monto_credito_maximo'] ? floatval($row['monto_credito_maximo']) : 0,
                    'cliente_saldo_credito_usado' => $row['saldo_credito_usado'] ? floatval($row['saldo_credito_usado']) : 0,
                    'usuario_id' => intval($row['usuario_id']),
                    'usuario_nombre' => $row['usuario_nombre'],
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'estatus_pago' => $row['estatus_pago'],
                    'tipo_entrega' => $row['tipo_entrega'],
                    'direccion_entrega' => $row['direccion_entrega'],
                    'estado' => $row['estado'],
                    'observaciones' => $row['observaciones'],
                    'total_productos' => intval($row['total_productos']),
                    'fecha_creacion' => $row['fecha_creacion'],
                    'fecha_actualizacion' => $row['fecha_actualizacion']
                ];
            }

            sendResponse(true, "Ventas obtenidas exitosamente", $ventas);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener ventas: " . $e->getMessage());
        }
        break;

    case 'getVenta':
        $id = intval($datapost['id'] ?? 0);

        if ($id <= 0) {
            sendResponse(false, "ID de venta requerido");
        }

        try {
            // Obtener datos de la venta
            $query = "SELECT
                v.id,
                v.folio,
                v.cliente_id,
                v.cliente_nombre,
                v.usuario_id,
                v.usuario_nombre,
                v.subtotal,
                v.impuesto,
                v.descuento,
                v.total,
                v.metodo_pago,
                v.estatus_pago,
                v.tipo_entrega,
                v.direccion_entrega,
                v.estado,
                v.observaciones,
                v.fecha_creacion,
                v.fecha_actualizacion
            FROM ventas v
            WHERE v.id = ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($row = $result->fetch_assoc()) {
                // Obtener detalles de la venta con información del almacén
                $detallesQuery = "SELECT
                    vd.id,
                    vd.producto_id,
                    vd.producto_nombre,
                    vd.cantidad,
                    vd.cantidad_escaneada,
                    vd.estatus_preparacion,
                    vd.precio_unitario,
                    vd.subtotal,
                    vd.impuesto,
                    vd.descuento,
                    vd.total,
                    p.almacen_id,
                    p.codigo_corto,
                    p.codigo_barras,
                    a.nombre as almacen_nombre
                FROM venta_detalles vd
                LEFT JOIN productos p ON vd.producto_id = p.id
                LEFT JOIN almacenes a ON p.almacen_id = a.id
                WHERE vd.venta_id = ?
                ORDER BY vd.id ASC";

                $detallesStmt = $conn->prepare($detallesQuery);
                $detallesStmt->bind_param("i", $id);
                $detallesStmt->execute();
                $detallesResult = $detallesStmt->get_result();

                $detalles = [];
                while ($detRow = $detallesResult->fetch_assoc()) {
                    $detalles[] = [
                        'id' => intval($detRow['id']),
                        'producto_id' => intval($detRow['producto_id']),
                        'producto_nombre' => $detRow['producto_nombre'],
                        'cantidad' => floatval($detRow['cantidad']),
                        'cantidad_escaneada' => intval($detRow['cantidad_escaneada']),
                        'estatus_preparacion' => $detRow['estatus_preparacion'],
                        'codigo_corto' => $detRow['codigo_corto'],
                        'codigo_barras' => $detRow['codigo_barras'],
                        'almacen_id' => $detRow['almacen_id'] ? intval($detRow['almacen_id']) : null,
                        'almacen_nombre' => $detRow['almacen_nombre'],
                        'precio_unitario' => floatval($detRow['precio_unitario']),
                        'subtotal' => floatval($detRow['subtotal']),
                        'impuesto' => floatval($detRow['impuesto']),
                        'descuento' => floatval($detRow['descuento']),
                        'total' => floatval($detRow['total'])
                    ];
                }

                $venta = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'cliente_id' => $row['cliente_id'] ? intval($row['cliente_id']) : null,
                    'cliente_nombre' => $row['cliente_nombre'] ?? 'Cliente General',
                    'usuario_id' => intval($row['usuario_id']),
                    'usuario_nombre' => $row['usuario_nombre'],
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'estatus_pago' => $row['estatus_pago'],
                    'tipo_entrega' => $row['tipo_entrega'],
                    'direccion_entrega' => $row['direccion_entrega'],
                    'estado' => $row['estado'],
                    'observaciones' => $row['observaciones'],
                    'detalles' => $detalles,
                    'fecha_creacion' => $row['fecha_creacion'],
                    'fecha_actualizacion' => $row['fecha_actualizacion']
                ];

                sendResponse(true, "Venta obtenida exitosamente", $venta);
            } else {
                sendResponse(false, "Venta no encontrada");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener venta: " . $e->getMessage());
        }
        break;

    case 'crearVenta':
        $cliente_id = isset($datapost['cliente_id']) && $datapost['cliente_id'] ? intval($datapost['cliente_id']) : null;
        $cliente_nombre = trim($datapost['cliente_nombre'] ?? 'Cliente General');
        $usuario_id = intval($datapost['usuario_id'] ?? 0);
        $usuario_nombre = trim($datapost['usuario_nombre'] ?? '');
        $metodo_pago = isset($datapost['metodo_pago']) ? $datapost['metodo_pago'] : null;  // Permitir NULL
        $estatus_pago = $datapost['estatus_pago'] ?? 'pendiente';
        $tipo_entrega = $datapost['tipo_entrega'] ?? 'pickup';
        $direccion_entrega = trim($datapost['direccion_entrega'] ?? '');
        $observaciones = trim($datapost['observaciones'] ?? '');
        $productos = $datapost['productos'] ?? [];
        $nombre_pc = isset($datapost['nombre_pc']) ? trim($datapost['nombre_pc']) : null;

        // Campos de comisión
        $aplica_comision = isset($datapost['aplica_comision']) ? intval($datapost['aplica_comision']) : 0;
        $comision_porcentaje = isset($datapost['comision_porcentaje']) ? floatval($datapost['comision_porcentaje']) : 0;
        $comision_monto = isset($datapost['comision_monto']) ? floatval($datapost['comision_monto']) : 0;

        // Obtener caja_id desde nombre_pc
        $caja_id = null;
        if ($nombre_pc) {
            $cajaStmt = $conn->prepare("SELECT id FROM cajas WHERE nombre_pc = ?");
            $cajaStmt->bind_param("s", $nombre_pc);
            $cajaStmt->execute();
            $cajaResult = $cajaStmt->get_result();
            if ($cajaRow = $cajaResult->fetch_assoc()) {
                $caja_id = intval($cajaRow['id']);
            }
        }

        // Validaciones
        if ($usuario_id <= 0) {
            sendResponse(false, "Usuario requerido");
        }

        if (empty($productos) || !is_array($productos)) {
            sendResponse(false, "Debe agregar al menos un producto");
        }

        // Validar crédito si el método de pago es crédito
        if ($metodo_pago === 'credito') {
            if (!$cliente_id) {
                sendResponse(false, "Se requiere un cliente para ventas a crédito");
            }

            // Verificar que el cliente tenga crédito activado
            $creditoQuery = "SELECT tiene_credito, monto_credito_maximo, saldo_credito_usado
                            FROM clientes WHERE id = ?";
            $creditoStmt = $conn->prepare($creditoQuery);
            $creditoStmt->bind_param("i", $cliente_id);
            $creditoStmt->execute();
            $creditoResult = $creditoStmt->get_result();

            if ($creditoResult->num_rows === 0) {
                sendResponse(false, "Cliente no encontrado");
            }

            $clienteData = $creditoResult->fetch_assoc();

            if (!$clienteData['tiene_credito']) {
                sendResponse(false, "El cliente no tiene crédito activado");
            }

            $credito_disponible = floatval($clienteData['monto_credito_maximo']) - floatval($clienteData['saldo_credito_usado']);

            if ($credito_disponible <= 0) {
                sendResponse(false, "El cliente no tiene crédito disponible");
            }
        }

        try {
            // Iniciar transacción
            $conn->begin_transaction();

            // Generar folio único
            $folio = 'VTA-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);

            // Verificar que el folio sea único
            $checkFolio = "SELECT id FROM ventas WHERE folio = ?";
            $checkStmt = $conn->prepare($checkFolio);
            $checkStmt->bind_param("s", $folio);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            while ($checkResult->num_rows > 0) {
                $folio = 'VTA-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
                $checkStmt->bind_param("s", $folio);
                $checkStmt->execute();
                $checkResult = $checkStmt->get_result();
            }

            // Calcular totales
            $subtotal = 0;
            $impuesto = 0;
            $descuento = 0;

            foreach ($productos as $prod) {
                $cantidad = floatval($prod['cantidad'] ?? 0);
                $precio = floatval($prod['precio_unitario'] ?? 0);
                $descProd = floatval($prod['descuento'] ?? 0);

                $subtotalProd = $cantidad * $precio;
                $subtotal += $subtotalProd;
                $descuento += $descProd;
            }

            $total = $subtotal - $descuento + $impuesto;

            // Validar que el total no exceda el crédito disponible (si es crédito)
            if ($metodo_pago === 'credito' && isset($credito_disponible)) {
                if ($total > $credito_disponible) {
                    $conn->rollback();
                    sendResponse(false, "El total de la venta (\$" . number_format($total, 2) . ") excede el crédito disponible (\$" . number_format($credito_disponible, 2) . ")");
                }
            }

            // Crear la venta
            // Calcular saldo_pendiente si es venta a crédito
            $saldo_pendiente = null;
            if ($metodo_pago === 'credito' && $estatus_pago === 'pendiente') {
                $saldo_pendiente = $total;
            }

            $insertVenta = "INSERT INTO ventas (
                folio, cliente_id, cliente_nombre, usuario_id, usuario_nombre, caja_id,
                subtotal, impuesto, descuento, total, metodo_pago, estatus_pago, saldo_pendiente, tipo_entrega, direccion_entrega, estado, observaciones,
                aplica_comision, comision_porcentaje, comision_monto
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, '', ?, ?, ?, ?)";

            $stmt = $conn->prepare($insertVenta);
            $stmt->bind_param(
                "sisisiidddssdsssids",
                $folio,
                $cliente_id,
                $cliente_nombre,
                $usuario_id,
                $usuario_nombre,
                $caja_id,
                $subtotal,
                $impuesto,
                $descuento,
                $total,
                $metodo_pago,
                $estatus_pago,
                $saldo_pendiente,
                $tipo_entrega,
                $direccion_entrega,
                $observaciones,
                $aplica_comision,
                $comision_porcentaje,
                $comision_monto
            );

            if (!$stmt->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al crear la venta");
            }

            $venta_id = $conn->insert_id;

            // Insertar detalles de la venta
            $insertDetalle = "INSERT INTO venta_detalles (
                venta_id, producto_id, producto_nombre, cantidad,
                precio_unitario, subtotal, impuesto, descuento, total,
                precio_tipo, aplica_comision
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

            $detalleStmt = $conn->prepare($insertDetalle);

            foreach ($productos as $prod) {
                $producto_id = intval($prod['producto_id']);
                $producto_nombre = trim($prod['producto_nombre']);
                $cantidad = floatval($prod['cantidad']);
                $precio_unitario = floatval($prod['precio_unitario']);
                $desc_prod = floatval($prod['descuento'] ?? 0);
                $imp_prod = floatval($prod['impuesto'] ?? 0);
                $precio_tipo_prod = isset($prod['precio_tipo']) ? trim($prod['precio_tipo']) : 'precio_lista';
                $aplica_comision_prod = isset($prod['aplica_comision']) ? intval($prod['aplica_comision']) : 0;

                $subtotal_prod = $cantidad * $precio_unitario;
                $total_prod = $subtotal_prod - $desc_prod + $imp_prod;

                $detalleStmt->bind_param(
                    "iisddddddsi",
                    $venta_id,
                    $producto_id,
                    $producto_nombre,
                    $cantidad,
                    $precio_unitario,
                    $subtotal_prod,
                    $imp_prod,
                    $desc_prod,
                    $total_prod,
                    $precio_tipo_prod,
                    $aplica_comision_prod
                );

                if (!$detalleStmt->execute()) {
                    $conn->rollback();
                    sendResponse(false, "Error al guardar detalle de producto");
                }

                // Actualizar stock del producto
                $updateStock = "UPDATE productos SET stock = stock - ? WHERE id = ?";
                $stockStmt = $conn->prepare($updateStock);
                $stockStmt->bind_param("di", $cantidad, $producto_id);
                $stockStmt->execute();
            }

            // Si es venta a crédito y el estatus es pendiente, actualizar saldo del cliente
            if ($metodo_pago === 'credito' && $estatus_pago === 'pendiente' && $cliente_id) {
                $updateCredito = "UPDATE clientes SET saldo_credito_usado = saldo_credito_usado + ? WHERE id = ?";
                $updateCreditoStmt = $conn->prepare($updateCredito);
                $updateCreditoStmt->bind_param("di", $total, $cliente_id);

                if (!$updateCreditoStmt->execute()) {
                    $conn->rollback();
                    sendResponse(false, "Error al actualizar saldo de crédito del cliente");
                }
            }

            // Confirmar transacción
            $conn->commit();
            sendResponse(true, "Venta creada exitosamente", ['id' => $venta_id, 'folio' => $folio]);
        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al crear venta: " . $e->getMessage());
        }
        break;

    case 'cancelarVenta':
        $id = intval($datapost['id'] ?? 0);
        $motivo = trim($datapost['motivo'] ?? '');

        if ($id <= 0) {
            sendResponse(false, "ID de venta requerido");
        }

        try {
            // Verificar que la venta existe y no está cancelada
            $checkQuery = "SELECT estado, observaciones FROM ventas WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows === 0) {
                sendResponse(false, "Venta no encontrada");
            }

            $venta = $checkResult->fetch_assoc();
            if ($venta['estado'] === 'cancelada') {
                sendResponse(false, "La venta ya está cancelada");
            }

            // Iniciar transacción
            $conn->begin_transaction();

            // Obtener detalles para devolver stock
            $detallesQuery = "SELECT producto_id, cantidad FROM venta_detalles WHERE venta_id = ?";
            $detallesStmt = $conn->prepare($detallesQuery);
            $detallesStmt->bind_param("i", $id);
            $detallesStmt->execute();
            $detallesResult = $detallesStmt->get_result();

            // Devolver stock
            while ($detalle = $detallesResult->fetch_assoc()) {
                $updateStock = "UPDATE productos SET stock = stock + ? WHERE id = ?";
                $stockStmt = $conn->prepare($updateStock);
                $cantidad = floatval($detalle['cantidad']);
                $producto_id = intval($detalle['producto_id']);
                $stockStmt->bind_param("di", $cantidad, $producto_id);
                $stockStmt->execute();
            }

            // Cancelar la venta
            $observaciones_nuevas = $venta['observaciones'];
            if (!empty($motivo)) {
                $observaciones_nuevas .= "\nCANCELADA: " . $motivo;
            }

            $updateQuery = "UPDATE ventas SET estado = 'cancelada', observaciones = ? WHERE id = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->bind_param("si", $observaciones_nuevas, $id);

            if ($updateStmt->execute()) {
                $conn->commit();
                sendResponse(true, "Venta cancelada exitosamente");
            } else {
                $conn->rollback();
                sendResponse(false, "Error al cancelar la venta");
            }
        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al cancelar venta: " . $e->getMessage());
        }
        break;

    case 'actualizarEstadoVenta':
        $id = intval($datapost['id'] ?? 0);
        $nuevo_estado = trim($datapost['estado'] ?? '');

        if ($id <= 0) {
            sendResponse(false, "ID de venta requerido");
        }

        if (empty($nuevo_estado)) {
            sendResponse(false, "Estado requerido");
        }

        try {
            // Verificar que la venta existe
            $checkQuery = "SELECT id FROM ventas WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows === 0) {
                sendResponse(false, "Venta no encontrada");
            }

            // Actualizar el estado
            $updateQuery = "UPDATE ventas SET estado = ? WHERE id = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->bind_param("si", $nuevo_estado, $id);

            if ($updateStmt->execute()) {
                sendResponse(true, "Estado actualizado exitosamente", ['id' => $id, 'estado' => $nuevo_estado]);
            } else {
                sendResponse(false, "Error al actualizar el estado");
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al actualizar estado: " . $e->getMessage());
        }
        break;

    case 'actualizarPagoVenta':
        $id = intval($datapost['id'] ?? 0);
        $metodo_pago = isset($datapost['metodo_pago']) ? $datapost['metodo_pago'] : null;  // Permitir NULL
        $estatus_pago = trim($datapost['estatus_pago'] ?? '');

        if ($id <= 0) {
            sendResponse(false, "ID de venta requerido");
        }

        if (empty($estatus_pago)) {
            sendResponse(false, "Estatus de pago requerido");
        }

        // Validar valores permitidos
        $metodos_validos = ['efectivo', 'tarjeta', 'transferencia', 'credito', 'mixto'];
        $estatus_validos = ['pagado', 'pendiente'];

        // Validar método de pago solo si no es null
        if ($metodo_pago !== null && !in_array($metodo_pago, $metodos_validos)) {
            sendResponse(false, "Método de pago no válido");
        }

        if (!in_array($estatus_pago, $estatus_validos)) {
            sendResponse(false, "Estatus de pago no válido");
        }

        try {
            // Iniciar transacción
            $conn->begin_transaction();

            // Obtener datos actuales de la venta
            $checkQuery = "SELECT cliente_id, metodo_pago, estatus_pago, total FROM ventas WHERE id = ?";
            $checkStmt = $conn->prepare($checkQuery);
            $checkStmt->bind_param("i", $id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows === 0) {
                $conn->rollback();
                sendResponse(false, "Venta no encontrada");
            }

            $ventaActual = $checkResult->fetch_assoc();
            $cliente_id = $ventaActual['cliente_id'];
            $metodo_anterior = $ventaActual['metodo_pago'];
            $estatus_anterior = $ventaActual['estatus_pago'];
            $total_venta = floatval($ventaActual['total']);

            // Calcular saldo_pendiente según el nuevo método de pago
            $nuevo_saldo_pendiente = null;
            if ($metodo_pago === 'credito' && $estatus_pago === 'pendiente') {
                $nuevo_saldo_pendiente = $total_venta;
            } elseif ($metodo_pago === 'credito' && $estatus_pago === 'pagado') {
                $nuevo_saldo_pendiente = 0;
            }

            // Actualizar el método, estatus de pago y saldo_pendiente
            $updateQuery = "UPDATE ventas SET metodo_pago = ?, estatus_pago = ?, saldo_pendiente = ? WHERE id = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->bind_param("ssdi", $metodo_pago, $estatus_pago, $nuevo_saldo_pendiente, $id);

            if (!$updateStmt->execute()) {
                $conn->rollback();
                sendResponse(false, "Error al actualizar la información de pago");
            }

            // Ajustar saldo de crédito del cliente si corresponde
            if ($cliente_id) {
                $ajuste_credito = 0;

                // Caso 1: Era crédito pendiente, ahora es crédito pagado -> REDUCIR saldo usado
                if ($metodo_anterior === 'credito' && $estatus_anterior === 'pendiente' &&
                    $metodo_pago === 'credito' && $estatus_pago === 'pagado') {
                    $ajuste_credito = -$total_venta; // Restar del saldo usado
                }

                // Caso 2: Era crédito pagado, ahora es crédito pendiente -> AUMENTAR saldo usado
                if ($metodo_anterior === 'credito' && $estatus_anterior === 'pagado' &&
                    $metodo_pago === 'credito' && $estatus_pago === 'pendiente') {
                    $ajuste_credito = $total_venta; // Sumar al saldo usado
                }

                // Caso 3: NO era crédito, ahora SÍ es crédito pendiente -> AUMENTAR saldo usado
                if ($metodo_anterior !== 'credito' && $metodo_pago === 'credito' && $estatus_pago === 'pendiente') {
                    $ajuste_credito = $total_venta;
                }

                // Caso 4: ERA crédito pendiente, ahora NO es crédito -> REDUCIR saldo usado
                if ($metodo_anterior === 'credito' && $estatus_anterior === 'pendiente' && $metodo_pago !== 'credito') {
                    $ajuste_credito = -$total_venta;
                }

                // Aplicar el ajuste si hay cambio
                if ($ajuste_credito != 0) {
                    $updateCreditoQuery = "UPDATE clientes SET saldo_credito_usado = saldo_credito_usado + ? WHERE id = ?";
                    $updateCreditoStmt = $conn->prepare($updateCreditoQuery);
                    $updateCreditoStmt->bind_param("di", $ajuste_credito, $cliente_id);

                    if (!$updateCreditoStmt->execute()) {
                        $conn->rollback();
                        sendResponse(false, "Error al actualizar saldo de crédito del cliente");
                    }
                }
            }

            // Confirmar transacción
            $conn->commit();
            sendResponse(true, "Información de pago actualizada exitosamente", [
                'id' => $id,
                'metodo_pago' => $metodo_pago,
                'estatus_pago' => $estatus_pago
            ]);
        } catch (Exception $e) {
            $conn->rollback();
            sendResponse(false, "Error al actualizar pago: " . $e->getMessage());
        }
        break;

    case 'actualizarEstatusPreparacion':
        $venta_detalle_id = intval($datapost['venta_detalle_id'] ?? 0);
        $incrementar_cantidad = $datapost['incrementar_cantidad'] ?? true;
        $completar_total = $datapost['completar_total'] ?? false;
        $estatus_preparacion = trim($datapost['estatus_preparacion'] ?? '');

        // Validaciones
        if ($venta_detalle_id <= 0) {
            sendResponse(false, "ID de detalle de venta requerido");
        }

        try {
            // Primero obtener la cantidad actual y cantidad escaneada
            $queryGet = "SELECT cantidad, cantidad_escaneada FROM venta_detalles WHERE id = ?";
            $stmtGet = $conn->prepare($queryGet);
            $stmtGet->bind_param("i", $venta_detalle_id);
            $stmtGet->execute();
            $result = $stmtGet->get_result();
            $detalle = $result->fetch_assoc();

            if (!$detalle) {
                sendResponse(false, "Detalle de venta no encontrado");
            }

            // Si se solicita completar el total (marcar como preparado)
            if ($completar_total) {
                // Establecer cantidad escaneada = cantidad total
                $nueva_cantidad_escaneada = $detalle['cantidad'];
                $nuevo_estatus = 'preparado';

                $query = "UPDATE venta_detalles
                         SET cantidad_escaneada = ?,
                             estatus_preparacion = ?
                         WHERE id = ?";

                $stmt = $conn->prepare($query);
                $stmt->bind_param("isi", $nueva_cantidad_escaneada, $nuevo_estatus, $venta_detalle_id);

                if ($stmt->execute()) {
                    sendResponse(true, "Producto marcado como preparado", [
                        'venta_detalle_id' => $venta_detalle_id,
                        'cantidad_escaneada' => $nueva_cantidad_escaneada,
                        'cantidad_total' => $detalle['cantidad'],
                        'estatus_preparacion' => $nuevo_estatus,
                        'completado' => true
                    ]);
                } else {
                    sendResponse(false, "Error al marcar como preparado: " . $stmt->error);
                }
            }
            // Si se solicita incrementar cantidad (escaneo de código de barras)
            else if ($incrementar_cantidad) {
                // Incrementar cantidad escaneada
                $nueva_cantidad_escaneada = $detalle['cantidad_escaneada'] + 1;

                // Determinar estatus según cantidad escaneada vs cantidad total
                $nuevo_estatus = ($nueva_cantidad_escaneada >= $detalle['cantidad']) ? 'preparado' : 'pendiente';

                // Actualizar
                $query = "UPDATE venta_detalles
                         SET cantidad_escaneada = ?,
                             estatus_preparacion = ?
                         WHERE id = ?";

                $stmt = $conn->prepare($query);
                $stmt->bind_param("isi", $nueva_cantidad_escaneada, $nuevo_estatus, $venta_detalle_id);

                if ($stmt->execute()) {
                    sendResponse(true, "Cantidad escaneada actualizada exitosamente", [
                        'venta_detalle_id' => $venta_detalle_id,
                        'cantidad_escaneada' => $nueva_cantidad_escaneada,
                        'cantidad_total' => $detalle['cantidad'],
                        'estatus_preparacion' => $nuevo_estatus,
                        'completado' => ($nuevo_estatus === 'preparado')
                    ]);
                } else {
                    sendResponse(false, "Error al actualizar cantidad escaneada: " . $stmt->error);
                }
            } else {
                // Actualización manual de estatus (sin incremento ni completar)
                if (!in_array($estatus_preparacion, ['pendiente', 'preparado'])) {
                    sendResponse(false, "Estatus de preparación inválido");
                }

                $query = "UPDATE venta_detalles
                         SET estatus_preparacion = ?
                         WHERE id = ?";

                $stmt = $conn->prepare($query);
                $stmt->bind_param("si", $estatus_preparacion, $venta_detalle_id);

                if ($stmt->execute()) {
                    sendResponse(true, "Estatus de preparación actualizado exitosamente", [
                        'venta_detalle_id' => $venta_detalle_id,
                        'estatus_preparacion' => $estatus_preparacion
                    ]);
                } else {
                    sendResponse(false, "Error al actualizar estatus de preparación: " . $stmt->error);
                }
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al actualizar estatus de preparación: " . $e->getMessage());
        }
        break;

    case 'reiniciarEscaneoProducto':
        $venta_detalle_id = intval($datapost['venta_detalle_id'] ?? 0);

        if ($venta_detalle_id <= 0) {
            sendResponse(false, "ID de detalle de venta requerido");
        }

        try {
            // Resetear cantidad escaneada a 0 y estatus a pendiente
            $query = "UPDATE venta_detalles
                     SET cantidad_escaneada = 0,
                         estatus_preparacion = 'pendiente'
                     WHERE id = ?";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $venta_detalle_id);

            if ($stmt->execute()) {
                sendResponse(true, "Escaneo reiniciado exitosamente", [
                    'venta_detalle_id' => $venta_detalle_id,
                    'cantidad_escaneada' => 0,
                    'estatus_preparacion' => 'pendiente'
                ]);
            } else {
                sendResponse(false, "Error al reiniciar escaneo: " . $stmt->error);
            }
        } catch (Exception $e) {
            sendResponse(false, "Error al reiniciar escaneo: " . $e->getMessage());
        }
        break;

    case 'getVentasDelDia':
        try {
            // Configurar zona horaria de México
            date_default_timezone_set('America/Mexico_City');

            $fecha_hoy = date('Y-m-d');


            $query = "SELECT
                v.id,
                v.folio,
                v.cliente_nombre,
                v.total,
                v.metodo_pago,
                v.estatus_pago,
                v.tipo_entrega,
                v.estado,
                v.fecha_creacion,
                COUNT(vd.id) as total_productos
            FROM ventas v
            LEFT JOIN venta_detalles vd ON v.id = vd.venta_id
            WHERE DATE(v.fecha_creacion) = ?
            GROUP BY v.id
            ORDER BY v.fecha_creacion DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("s", $fecha_hoy);
            $stmt->execute();
            $result = $stmt->get_result();

            $ventas = [];
            while ($row = $result->fetch_assoc()) {
                $ventas[] = [
                    'id' => intval($row['id']),
                    'folio' => $row['folio'],
                    'cliente_nombre' => $row['cliente_nombre'] ?? 'Cliente General',
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'estatus_pago' => $row['estatus_pago'],
                    'tipo_entrega' => $row['tipo_entrega'],
                    'estado' => $row['estado'],
                    'total_productos' => intval($row['total_productos']),
                    'fecha_creacion' => $row['fecha_creacion']
                ];
            }

            sendResponse(true, "Ventas del día obtenidas exitosamente", $ventas);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener ventas del día: " . $e->getMessage());
        }
        break;

    case 'getVentasPorEstadoYAlmacen':
        $estado = $datapost['estado'] ?? '';
        $almacen_id = intval($datapost['almacen_id'] ?? 0);

        if (empty($estado)) {
            sendResponse(false, "Estado requerido");
        }

        if ($almacen_id <= 0) {
            sendResponse(false, "ID de almacén requerido");
        }

        try {
            // Obtener ventas con sus detalles filtradas por estado y almacén
            // El almacen_id está en la tabla productos, no en ventas
            $query = "SELECT DISTINCT
                v.id,
                v.folio,
                v.cliente_id,
                v.cliente_nombre,
                v.usuario_id,
                v.usuario_nombre,
                v.subtotal,
                v.impuesto,
                v.descuento,
                v.total,
                v.metodo_pago,
                v.tipo_entrega,
                v.direccion_entrega,
                v.estado,
                v.observaciones,
                v.fecha_creacion,
                v.fecha_actualizacion
            FROM ventas v
            INNER JOIN venta_detalles vd ON v.id = vd.venta_id
            INNER JOIN productos p ON vd.producto_id = p.id
            WHERE v.estado = ? AND p.almacen_id = ?
            ORDER BY v.fecha_creacion DESC";

            $stmt = $conn->prepare($query);
            $stmt->bind_param("si", $estado, $almacen_id);
            $stmt->execute();
            $result = $stmt->get_result();

            $ventas = [];
            while ($row = $result->fetch_assoc()) {
                $venta_id = intval($row['id']);

                // Obtener detalles de esta venta solo para productos del almacén especificado
                $detallesQuery = "SELECT
                    vd.id,
                    vd.producto_id,
                    p.nombre as producto_nombre,
                    p.codigo_barras,
                    vd.cantidad,
                    vd.precio_unitario,
                    vd.subtotal,
                    vd.impuesto,
                    vd.descuento,
                    vd.total
                FROM venta_detalles vd
                INNER JOIN productos p ON vd.producto_id = p.id
                WHERE vd.venta_id = ? AND p.almacen_id = ?";

                $stmtDetalles = $conn->prepare($detallesQuery);
                $stmtDetalles->bind_param("ii", $venta_id, $almacen_id);
                $stmtDetalles->execute();
                $resultDetalles = $stmtDetalles->get_result();

                $detalles = [];
                while ($detalle = $resultDetalles->fetch_assoc()) {
                    $detalles[] = [
                        'id' => intval($detalle['id']),
                        'producto_id' => intval($detalle['producto_id']),
                        'producto_nombre' => $detalle['producto_nombre'],
                        'codigo_barras' => $detalle['codigo_barras'],
                        'cantidad' => floatval($detalle['cantidad']),
                        'precio_unitario' => floatval($detalle['precio_unitario']),
                        'subtotal' => floatval($detalle['subtotal']),
                        'impuesto' => floatval($detalle['impuesto']),
                        'descuento' => floatval($detalle['descuento']),
                        'total' => floatval($detalle['total'])
                    ];
                }

                $ventas[] = [
                    'id' => $venta_id,
                    'folio' => $row['folio'],
                    'cliente_id' => $row['cliente_id'] ? intval($row['cliente_id']) : null,
                    'cliente_nombre' => $row['cliente_nombre'] ?? 'Cliente General',
                    'usuario_id' => intval($row['usuario_id']),
                    'usuario_nombre' => $row['usuario_nombre'],
                    'subtotal' => floatval($row['subtotal']),
                    'impuesto' => floatval($row['impuesto']),
                    'descuento' => floatval($row['descuento']),
                    'total' => floatval($row['total']),
                    'metodo_pago' => $row['metodo_pago'],
                    'tipo_entrega' => $row['tipo_entrega'],
                    'direccion_entrega' => $row['direccion_entrega'],
                    'estado' => $row['estado'],
                    'observaciones' => $row['observaciones'],
                    'total_productos' => count($detalles), // Contar solo productos del almacén
                    'fecha_creacion' => $row['fecha_creacion'],
                    'fecha_actualizacion' => $row['fecha_actualizacion'],
                    'detalles' => $detalles
                ];
            }

            sendResponse(true, "Ventas obtenidas exitosamente", $ventas);
        } catch (Exception $e) {
            sendResponse(false, "Error al obtener ventas: " . $e->getMessage());
        }
        break;

    default:
        sendResponse(false, "Operación no válida para ventas");
        break;
}
