<?php
/**
 * Configuración de sincronización BD Local ↔ BD Nube
 *
 * Define las tablas, orden de dependencias FK, modos de sync,
 * y configuración de conexión.
 */
if (!defined('BASEPATH')) {
    die('Acceso directo no permitido');
}

// -----------------------------------------------
// CONFIGURACIÓN DE CONEXIÓN NUBE
// -----------------------------------------------

function getSyncConfig($conn) {
    $config = [
        'cloud_url'  => '',
        'negocio'    => '',
        'sucursal'   => '',
        'batch_size' => 500,
        'timeout'    => 30,
    ];

    // Leer configuración desde BD
    $keys = ['caja_api_nube_url', 'caja_api_nube_negocio', 'caja_api_nube_sucursal'];
    $placeholders = implode(',', array_fill(0, count($keys), '?'));
    $stmt = $conn->prepare("SELECT clave, valor FROM configuraciones WHERE clave IN ($placeholders)");
    $types = str_repeat('s', count($keys));
    $stmt->bind_param($types, ...$keys);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        switch ($row['clave']) {
            case 'caja_api_nube_url':
                $config['cloud_url'] = $row['valor'];
                break;
            case 'caja_api_nube_negocio':
                $config['negocio'] = $row['valor'];
                break;
            case 'caja_api_nube_sucursal':
                $config['sucursal'] = $row['valor'];
                break;
        }
    }
    $stmt->close();

    return $config;
}

// -----------------------------------------------
// DEFINICIÓN DE TABLAS PARA SINCRONIZACIÓN
// -----------------------------------------------
// Cada tabla define:
//   pk          - Nombre de la columna Primary Key
//   timestamp   - Columna para detectar cambios (updated_at o fecha_actualizacion)
//   fk          - Array de [columna_fk => tabla_referenciada]
//   mode        - 'bidirectional' (ambas sucursales) o 'owned' (solo la sucursal dueña)
//   level       - Nivel topológico (0=sin dependencias, mayor=más dependencias)
//   exclude_cols- Columnas que no se sincronizan (opcionales, como passwords)

$SYNC_TABLES = [

    // === NIVEL 0: Sin dependencias FK ===
    'categorias' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],
    'almacenes' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],
    'configuraciones' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'owned',
        'level' => 0,
    ],
    'cajas' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'owned',
        'level' => 0,
    ],
    'cat_regimen_fiscal' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],
    'cat_uso_cfdi' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],
    'cat_forma_pago' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],
    'cat_metodo_pago' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],
    'precio_rangos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 0,
    ],

    // === NIVEL 1: Dependen solo de catálogos base ===
    'roles' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 1,
    ],
    'proveedores' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 1,
    ],
    'vehiculos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 1,
    ],
    'rangos_flete' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 1,
    ],

    // === NIVEL 2: Dependen de nivel 0/1 ===
    'permisos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 2,
    ],
    'usuarios' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['rol_id' => 'roles'],
        'mode' => 'bidirectional',
        'level' => 2,
        'exclude_cols' => ['password'],
    ],
    'productos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['categoria_id' => 'categorias'],
        'mode' => 'bidirectional',
        'level' => 2,
    ],
    'clientes' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 2,
    ],

    // === NIVEL 3: Tablas de relación/detalle de nivel 2 ===
    'rol_permisos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['rol_id' => 'roles', 'permiso_id' => 'permisos'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'usuario_almacenes' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios', 'almacen_id' => 'almacenes'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'producto_equivalencias' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['producto_id' => 'productos', 'producto_equivalente_id' => 'productos'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'descuentos_volumen' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['producto_id' => 'productos'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'cliente_telefonos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['cliente_id' => 'clientes'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'cliente_direcciones' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['cliente_id' => 'clientes'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'cliente_datos_fiscales' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['cliente_id' => 'clientes'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'configuracion_rangos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => [],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'vehiculo_rangos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['vehiculo_id' => 'vehiculos'],
        'mode' => 'bidirectional',
        'level' => 3,
    ],
    'diferencia_costos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['producto_id' => 'productos'],
        'mode' => 'owned',
        'level' => 3,
    ],

    // === NIVEL 4: Transacciones principales ===
    'ventas' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['cliente_id' => 'clientes', 'usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 4,
    ],
    'ventas_mostrador' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios', 'caja_id' => 'cajas'],
        'mode' => 'owned',
        'level' => 4,
    ],
    'ordenes_compra' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['proveedor_id' => 'proveedores', 'usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 4,
    ],
    'rutas_entrega' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['vehiculo_id' => 'vehiculos', 'usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 4,
    ],
    'sesiones' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 4,
    ],
    'log_actividad' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 4,
    ],
    'log_seguridad' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 4,
    ],

    // === NIVEL 5: Detalles de transacciones ===
    'venta_detalles' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['venta_id' => 'ventas', 'producto_id' => 'productos'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'venta_pagos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['venta_id' => 'ventas'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'ventas_mostrador_detalles' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['venta_mostrador_id' => 'ventas_mostrador', 'producto_id' => 'productos'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'ventas_mostrador_pagos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['venta_mostrador_id' => 'ventas_mostrador'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'cortes_caja' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios', 'caja_id' => 'cajas'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'retiros_efectivo' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios', 'caja_id' => 'cajas'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'ingresos_efectivo' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['usuario_id' => 'usuarios', 'caja_id' => 'cajas'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'orden_compra_detalle' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['orden_compra_id' => 'ordenes_compra', 'producto_id' => 'productos'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'ruta_pedidos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['ruta_id' => 'rutas_entrega', 'venta_id' => 'ventas'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'pagos_clientes' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['venta_id' => 'ventas', 'cliente_id' => 'clientes', 'usuario_id' => 'usuarios'],
        'mode' => 'owned',
        'level' => 5,
    ],
    'facturas' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['cliente_id' => 'clientes'],
        'mode' => 'owned',
        'level' => 5,
    ],

    // === NIVEL 6: Detalles de facturas ===
    'factura_conceptos' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['factura_id' => 'facturas'],
        'mode' => 'owned',
        'level' => 6,
    ],
    'factura_ventas' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['factura_id' => 'facturas', 'venta_id' => 'ventas'],
        'mode' => 'owned',
        'level' => 6,
    ],
    'factura_ventas_mostrador' => [
        'pk' => 'id',
        'timestamp' => 'updated_at',
        'fk' => ['factura_id' => 'facturas', 'venta_mostrador_id' => 'ventas_mostrador'],
        'mode' => 'owned',
        'level' => 6,
    ],
];

/**
 * Retorna las tablas ordenadas por nivel topológico (para push/pull seguro)
 */
function getTablesInOrder($syncTables) {
    $ordered = $syncTables;
    uasort($ordered, function($a, $b) {
        return $a['level'] - $b['level'];
    });
    return $ordered;
}

/**
 * Retorna solo los nombres de tabla en orden
 */
function getTableNamesInOrder($syncTables) {
    $ordered = getTablesInOrder($syncTables);
    return array_keys($ordered);
}
