<?php
/**
 * Endpoints del lado NUBE para sincronización
 *
 * - syncReceive: Recibe datos de una sucursal (push de la sucursal)
 * - syncSend:    Envía datos a una sucursal (pull de la sucursal)
 *
 * Este archivo se ejecuta en el servidor de la NUBE.
 * La BD nube tiene columnas extra: sucursal, id_origen en cada tabla.
 */
if (!defined('BASEPATH')) {
    die('Acceso directo no permitido');
}

require_once(__DIR__ . '/sync_config.php');
require_once(__DIR__ . '/sync_utils.php');

switch ($tipo) {
    case 'syncReceive':
        handleSyncReceive($conn, $datapost);
        break;

    case 'syncSend':
        handleSyncSend($conn, $datapost);
        break;

    default:
        sendResponse(false, "Operación no reconocida: $tipo");
}

// =============================================
// syncReceive: Nube recibe datos de una sucursal
// =============================================

/**
 * Recibe un lote de registros de una sucursal y hace UPSERT en la BD nube.
 * Dedup por (sucursal, id_origen).
 *
 * Payload esperado:
 * {
 *   negocio: string,
 *   sucursal: string,
 *   tabla: string,
 *   registros: [ { ...campos... }, ... ]
 * }
 */
function handleSyncReceive($conn, $datapost) {
    global $SYNC_TABLES;

    $negocio   = $datapost['negocio'] ?? '';
    $sucursal  = $datapost['sucursal'] ?? '';
    $tableName = $datapost['tabla'] ?? '';
    $registros = $datapost['registros'] ?? [];

    if (empty($sucursal) || empty($tableName)) {
        sendResponse(false, "Parámetros requeridos: sucursal, tabla");
    }

    if (!isset($SYNC_TABLES[$tableName])) {
        sendResponse(false, "Tabla no válida para sync: $tableName");
    }

    if (empty($registros)) {
        sendResponse(true, "Sin registros para procesar", ['insertados' => 0, 'actualizados' => 0]);
    }

    $tableConfig = $SYNC_TABLES[$tableName];
    $pk = $tableConfig['pk'];
    $tsCol = $tableConfig['timestamp'];
    $startTime = microtime(true);

    $logId = logSync($conn, [
        'tabla'     => $tableName,
        'sucursal'  => $sucursal,
        'direccion' => 'push',  // Desde perspectiva del log, la sucursal hizo push
        'estado'    => 'iniciado',
    ]);

    $insertados = 0;
    $actualizados = 0;
    $conflictos = 0;
    $errores = [];

    $conn->begin_transaction();

    try {
        foreach ($registros as $idx => $registro) {
            $idOrigen = $registro[$pk] ?? null;
            if ($idOrigen === null) {
                $errores[] = "Registro #$idx sin PK ($pk)";
                continue;
            }

            // Verificar si ya existe en la nube por (sucursal, id_origen)
            $existing = getCloudRecord($conn, $tableName, $sucursal, $idOrigen, $tsCol);

            if ($existing) {
                // Verificar conflicto: ¿fue modificado por otra fuente desde el último sync?
                $remoteTimestamp = $registro[$tsCol] ?? null;
                $cloudTimestamp = $existing[$tsCol] ?? null;

                if ($cloudTimestamp && $remoteTimestamp) {
                    $winner = resolveConflict($cloudTimestamp, $remoteTimestamp, '__cloud__', $sucursal);
                    if ($winner === 'local') {
                        // La nube tiene datos más recientes, no sobrescribir
                        $conflictos++;
                        continue;
                    }
                }

                // Actualizar registro existente
                $updateData = $registro;
                unset($updateData[$pk]); // No actualizar PK de nube
                $excludeCols = $tableConfig['exclude_cols'] ?? [];
                foreach ($excludeCols as $exc) {
                    unset($updateData[$exc]);
                }

                updateCloudRecord($conn, $tableName, $sucursal, $idOrigen, $updateData);
                $actualizados++;
            } else {
                // Insertar nuevo registro
                $insertData = $registro;
                $originalId = $insertData[$pk];
                unset($insertData[$pk]); // La nube genera su propio ID

                $excludeCols = $tableConfig['exclude_cols'] ?? [];
                foreach ($excludeCols as $exc) {
                    unset($insertData[$exc]);
                }

                // Agregar campos de nube
                $insertData['sucursal'] = $sucursal;
                $insertData['id_origen'] = $originalId;

                insertCloudRecord($conn, $tableName, $insertData);
                $insertados++;
            }
        }

        $conn->commit();

        $duracion = round((microtime(true) - $startTime) * 1000);

        updateSyncLog($conn, $logId, [
            'registros_recibidos'  => $insertados + $actualizados,
            'registros_conflictos' => $conflictos,
            'estado'               => 'completado',
            'duracion_ms'          => $duracion,
        ]);

        sendResponse(true, "Recibidos correctamente", [
            'insertados'  => $insertados,
            'actualizados' => $actualizados,
            'conflictos'  => $conflictos,
            'errores'     => $errores,
            'duracion_ms' => $duracion,
        ]);

    } catch (Exception $e) {
        $conn->rollback();

        $duracion = round((microtime(true) - $startTime) * 1000);

        updateSyncLog($conn, $logId, [
            'estado'        => 'error',
            'error_mensaje' => $e->getMessage(),
            'duracion_ms'   => $duracion,
        ]);

        sendResponse(false, "Error procesando datos: " . $e->getMessage());
    }
}

// =============================================
// syncSend: Nube envía datos a una sucursal
// =============================================

/**
 * Envía registros de OTRAS sucursales a la sucursal solicitante.
 * Solo envía registros que la sucursal NO ha visto (cambiados después de 'since').
 *
 * Payload esperado:
 * {
 *   negocio: string,
 *   sucursal: string,       // Sucursal que pide datos (excluida de resultados)
 *   tabla: string,
 *   since: datetime|null,   // Desde cuándo
 *   page: int,
 *   batch_size: int
 * }
 */
function handleSyncSend($conn, $datapost) {
    global $SYNC_TABLES;

    $negocio    = $datapost['negocio'] ?? '';
    $sucursal   = $datapost['sucursal'] ?? '';
    $tableName  = $datapost['tabla'] ?? '';
    $since      = $datapost['since'] ?? null;
    $page       = (int)($datapost['page'] ?? 0);
    $batchSize  = (int)($datapost['batch_size'] ?? 500);

    if (empty($sucursal) || empty($tableName)) {
        sendResponse(false, "Parámetros requeridos: sucursal, tabla");
    }

    if (!isset($SYNC_TABLES[$tableName])) {
        sendResponse(false, "Tabla no válida para sync: $tableName");
    }

    $tableConfig = $SYNC_TABLES[$tableName];
    $tsCol = $tableConfig['timestamp'];
    $offset = $page * $batchSize;

    try {
        // Obtener columnas de la tabla (incluyendo sucursal e id_origen)
        $columns = getTableColumns($conn, $tableName, $tableConfig['exclude_cols'] ?? []);
        $colList = implode(', ', array_map(function($c) { return "`$c`"; }, $columns));

        // Construir query: registros de OTRAS sucursales, modificados después de 'since'
        $where = "sucursal != ? AND sucursal != ''";
        $params = [$sucursal];
        $types = 's';

        if ($since) {
            $where .= " AND `$tsCol` > ?";
            $params[] = $since;
            $types .= 's';
        }

        $sql = "SELECT $colList FROM `$tableName` WHERE $where ORDER BY `$tsCol` ASC LIMIT ? OFFSET ?";
        $params[] = $batchSize;
        $params[] = $offset;
        $types .= 'ii';

        $stmt = $conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();

        $records = [];
        while ($row = $result->fetch_assoc()) {
            $records[] = $row;
        }
        $stmt->close();

        $hasMore = (count($records) === $batchSize);

        sendResponse(true, "Datos enviados", [
            'records'  => $records,
            'has_more' => $hasMore,
            'page'     => $page,
            'count'    => count($records),
        ]);

    } catch (Exception $e) {
        sendResponse(false, "Error obteniendo datos: " . $e->getMessage());
    }
}

// =============================================
// FUNCIONES AUXILIARES PARA BD NUBE
// =============================================

/**
 * Busca un registro en la nube por (sucursal, id_origen)
 */
function getCloudRecord($conn, $table, $sucursal, $idOrigen, $tsCol) {
    $stmt = $conn->prepare(
        "SELECT id, `$tsCol` FROM `$table` WHERE sucursal = ? AND id_origen = ?"
    );
    $stmt->bind_param('si', $sucursal, $idOrigen);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    return $row;
}

/**
 * Actualiza un registro en la nube por (sucursal, id_origen)
 */
function updateCloudRecord($conn, $table, $sucursal, $idOrigen, $data) {
    if (empty($data)) return;

    $sets = [];
    $params = [];
    $types = '';

    foreach ($data as $col => $val) {
        $sets[] = "`$col` = ?";
        $params[] = $val;
        if (is_null($val)) {
            $types .= 's';
        } elseif (is_int($val)) {
            $types .= 'i';
        } elseif (is_float($val)) {
            $types .= 'd';
        } else {
            $types .= 's';
        }
    }

    $params[] = $sucursal;
    $params[] = $idOrigen;
    $types .= 'si';

    $sql = "UPDATE `$table` SET " . implode(', ', $sets) . " WHERE sucursal = ? AND id_origen = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $stmt->close();
}

/**
 * Inserta un registro nuevo en la nube
 */
function insertCloudRecord($conn, $table, $data) {
    if (empty($data)) return false;

    $columns = array_keys($data);
    $colList = implode(', ', array_map(function($c) { return "`$c`"; }, $columns));
    $placeholders = implode(', ', array_fill(0, count($columns), '?'));

    $params = array_values($data);
    $types = '';
    foreach ($params as $val) {
        if (is_null($val)) {
            $types .= 's';
        } elseif (is_int($val)) {
            $types .= 'i';
        } elseif (is_float($val)) {
            $types .= 'd';
        } else {
            $types .= 's';
        }
    }

    $sql = "INSERT INTO `$table` ($colList) VALUES ($placeholders)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $id = $stmt->insert_id;
    $stmt->close();

    return $id;
}
