<?php
// Funciones generales para POS Abarrotes

function getCode($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return $randomString;
}

function fechaYMDaDMY($fecha) {
    if (empty($fecha)) return '';
    $date = new DateTime($fecha);
    return $date->format('d/m/Y');
}

function fechaYMDHISaDMYHI($fecha) {
    if (empty($fecha)) return '';
    $date = new DateTime($fecha);
    return $date->format('d/m/Y H:i');
}

function formatearPrecio($precio) {
    return '$' . number_format($precio, 2, '.', ',');
}

function generarFolio($prefijo = 'POS') {
    return $prefijo . date('Ymd') . rand(1000, 9999);
}

function validarEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function validarTelefono($telefono) {
    // Remover espacios y caracteres especiales
    $telefono = preg_replace('/[^0-9]/', '', $telefono);
    // Validar longitud (10 dígitos para México)
    return strlen($telefono) == 10;
}

function limpiarTexto($texto) {
    return trim(strip_tags($texto));
}

function validarCodigoBarras($codigo) {
    // Validar que solo contenga números y tenga longitud apropiada
    return preg_match('/^\d{8,13}$/', $codigo);
}

function calcularImpuesto($subtotal, $tasa = 0.16) {
    return $subtotal * $tasa;
}

function calcularTotal($subtotal, $impuesto = 0, $descuento = 0) {
    return $subtotal + $impuesto - $descuento;
}

function formatearCodigoBarras($codigo) {
    // Formatear código de barras para mejor legibilidad
    if (strlen($codigo) == 13) {
        return substr($codigo, 0, 1) . '-' . substr($codigo, 1, 6) . '-' . substr($codigo, 7, 6);
    }
    return $codigo;
}

function getEstadoStock($cantidad, $minimo = 5, $critico = 2) {
    if ($cantidad <= $critico) return 'critico';
    if ($cantidad <= $minimo) return 'bajo';
    return 'normal';
}

function validarRFC($rfc) {
    // Validación básica de RFC mexicano
    $pattern = '/^[A-ZÑ&]{3,4}[0-9]{6}[A-V1-9][A-Z1-9][0-9A]$/';
    return preg_match($pattern, strtoupper($rfc));
}

function generarPassword($length = 8) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*';
    return substr(str_shuffle($characters), 0, $length);
}

function logActivity($accion, $usuario_id, $detalles = '') {
    global $conn;
    $query = "INSERT INTO log_actividad (usuario_id, accion, detalles, fecha) VALUES (?, ?, ?, NOW())";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("iss", $usuario_id, $accion, $detalles);
    return $stmt->execute();
}

function verificarPermiso($usuario_id, $permiso) {
    global $conn;
    $query = "SELECT COUNT(*) as count FROM usuarios u
              JOIN roles r ON u.rol_id = r.id
              JOIN permisos_rol pr ON r.id = pr.rol_id
              JOIN permisos p ON pr.permiso_id = p.id
              WHERE u.id = ? AND p.nombre = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("is", $usuario_id, $permiso);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    return $row['count'] > 0;
}

function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

function generarToken($user_id) {
    $payload = [
        'user_id' => $user_id,
        'timestamp' => time(),
        'random' => getCode(10)
    ];
    return base64_encode(json_encode($payload));
}

function validarToken($token) {
    try {
        $payload = json_decode(base64_decode($token), true);
        // Token válido por 24 horas
        return (time() - $payload['timestamp']) < 86400;
    } catch (Exception $e) {
        return false;
    }
}

function enviarEmailNotificacion($to, $subject, $message) {
    // Configurar y enviar email (implementar según necesidades)
    return mail($to, $subject, $message);
}

function getConfiguracion($clave, $default = null) {
    global $conn;
    $query = "SELECT valor FROM configuracion WHERE clave = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $clave);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($row = $result->fetch_assoc()) {
        return $row['valor'];
    }
    return $default;
}

function setConfiguracion($clave, $valor) {
    global $conn;
    $query = "INSERT INTO configuracion (clave, valor) VALUES (?, ?)
              ON DUPLICATE KEY UPDATE valor = VALUES(valor)";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("ss", $clave, $valor);
    return $stmt->execute();
}
?>